package com.tandbergtv.watchpoint.studio.ui.properties;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.draw2d.ColorConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.NodeElement;

import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.model.AutomaticTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.IHumanResourceTypeElement;
import com.tandbergtv.watchpoint.studio.ui.model.IResourceTypeElement;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.ResourceGroup;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * Composite UI for showing the resource group section.
 * 
 * @author Imran Naqvi
 * 
 */
public class ResourceGroupElementConfigurationComposite implements SelectionListener {

	private TabbedPropertySheetWidgetFactory widgetFactory;

	private Composite parent;

	private CCombo resourceGroupCombo;

	private Label resourceGroupLabel;

	private IResourceTypeElement resourceTypeElement;
	private String messageUID;

	private ResourceGroup resourceGroup;

	private ResourceGroupElementConfigurationComposite() {
	}

	/**
	 * Creates a ResourceGroupElementConfigurationComposite object in the parent object using the
	 * given widget factory.
	 * 
	 * @param widgetFactory
	 * @param parent
	 * @return
	 */
	public static ResourceGroupElementConfigurationComposite create(
			TabbedPropertySheetWidgetFactory widgetFactory, Composite parent) {
		ResourceGroupElementConfigurationComposite result = new ResourceGroupElementConfigurationComposite();
		result.widgetFactory = widgetFactory;
		result.parent = parent;
		result.create();
		return result;
	}

	/**
	 * Sets the Resource group element and updates controls.
	 * 
	 * @param resourceTypeElement
	 */
	public void setResourceGroupElement(IResourceTypeElement resourceTypeElement) {
		this.resourceTypeElement = resourceTypeElement;
		/* Disable combo box if the node definition's internal node hasnt been set yet */
		if (this.resourceTypeElement instanceof NodeDefinition) {
			NodeDefinition nodeDefinition = (NodeDefinition) this.resourceTypeElement;
			setMessageUID(nodeDefinition);
			if (nodeDefinition.getNode() == null) {
				this.resourceGroupCombo.setEnabled(false);
			} else {
				this.resourceGroupCombo.setEnabled(true);
			}
		}
		unhookSelectionListener();
		if (resourceTypeElement != null) {
			updateControls();
			hookSelectionListener();
			fillResourceGroupCombo();
		}
	}
	
	private void setMessageUID(NodeDefinition rootNode) {
		if (rootNode.getNode() instanceof AutomaticTaskNode) {
			this.messageUID = rootNode.getUid();
		} else if (rootNode.getNode() instanceof NodeGroup) {
			NodeGroup group = (NodeGroup) rootNode.getNode();
			for (NodeElement ssElement : group.getNodeElements()) {
				if (ssElement instanceof NodeDefinition) {
					NodeDefinition nodeDefinition = (NodeDefinition) ssElement;
					this.messageUID = nodeDefinition.getUid();
				}
			}
		}
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {
		if (e.widget == resourceGroupCombo) {
			if (resourceGroup == null) {
				resourceGroup = (ResourceGroup) this.resourceTypeElement.getFactory().createById(
						SemanticElementConstants.RESOURCEGROUP_SEID);
			}
			resourceGroup.setName(resourceGroupCombo.getItem(resourceGroupCombo.getSelectionIndex()));
			resourceTypeElement.setResourceGroup(resourceGroup);
		}
	}

	/*
	 * Hooks the selection listeners.
	 */
	private void hookSelectionListener() {
		resourceGroupCombo.addSelectionListener(this);
	}

	/*
	 * Unhooks the selection listeners.
	 * 
	 */
	private void unhookSelectionListener() {
		resourceGroupCombo.removeSelectionListener(this);
	}

	/*
	 * Updates the controls.
	 * 
	 */
	private void updateControls() {
		resourceGroup = resourceTypeElement.getResourceGroup();
	}

	/*
	 * Creates the widgets for this composite.
	 * 
	 */
	private void create() {
		Composite composite = widgetFactory.createFlatFormComposite(parent);
		resourceGroupLabel = new Label(composite, SWT.NONE);// widgetFactory.createLabel(parent,
		// "Resource Group:");
		resourceGroupLabel.setText("Resource Group");
		resourceGroupLabel.setLayoutData(createResourceGroupLabelLayoutData());
		resourceGroupLabel.setBackground(ColorConstants.white);
		resourceGroupLabel.pack();

		resourceGroupCombo = widgetFactory.createCCombo(composite, SWT.DROP_DOWN | SWT.SINGLE
				| SWT.V_SCROLL | SWT.H_SCROLL | SWT.READ_ONLY);
		resourceGroupCombo.setLayoutData(createResourceGroupComboLayoutData());
		if (this.resourceTypeElement instanceof NodeDefinition) {
			String defName = ((NodeDefinition) this.resourceTypeElement).getDefinitionName();
			if (defName == null || defName.trim().length() == 0)
				resourceGroupCombo.setEnabled(false);
		}

		resourceGroupCombo.pack();

	}

	/*
	 * Creates layout data for the resource group label.
	 * 
	 * @return
	 */
	private FormData createResourceGroupLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0,-8);
		data.top = new FormAttachment(0, 2);
		return data;
	}

	/*
	 * Creates layout data for the resource group combo.
	 * 
	 * @return
	 */
	private FormData createResourceGroupComboLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 99);
		data.top = new FormAttachment(0, 0);
		data.width = 200;
		return data;
	}

	/*
	 * Fills the resource group combo with the resource groups in the database.
	 * 
	 */
	private void fillResourceGroupCombo() {
		this.resourceGroupCombo.removeAll();
		List<com.tandbergtv.watchpoint.studio.dto.ResourceGroup> groups = new ArrayList<com.tandbergtv.watchpoint.studio.dto.ResourceGroup>();

		if (this.resourceTypeElement instanceof IHumanResourceTypeElement) {
			groups = ServiceFactory.createFactory()
					.createResourceGroupService()
					.getResourceGroupsByHumanResourceType();
		} else {
			if (messageUID != null && !messageUID.isEmpty()) {
				groups = ServiceFactory.createFactory()
						.createResourceGroupService()
						.getResourceGroupsByResourceType(Utility.getSystemIDByMessageUID(messageUID));
			}
		}
		
		int index = 0;
		boolean found = false;
		for (com.tandbergtv.watchpoint.studio.dto.ResourceGroup group : groups) {
			this.resourceGroupCombo.add(group.getName());
			
			if (this.resourceTypeElement.getResourceGroup() != null 
			    && group.getName().equals(this.resourceTypeElement.getResourceGroup().getName()) ) {
				this.resourceGroupCombo.select(index);
				found = true;
			}
			
			index++;
		}
		
		// means that the resource group present in the XML file is not present in the 
		// resource group database. should add the resource group set in the XML file and select it.
		if (!found && this.resourceTypeElement.getResourceGroup() != null) {
			this.resourceGroupCombo.add(this.resourceTypeElement.getResourceGroup().getName());
			this.resourceGroupCombo.select(index);
		}
	}
}
