package com.tandbergtv.watchpoint.studio.util;

import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.INVALID_BRACKETS;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.INVALID_CONDITION_EXPRESSION;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.INVALID_DECISION_EXPRESSION;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.INVALID_NUMBER_OF_VARIABLES;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.INVALID_OPEN_CLOSE_PARENTHESIS;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.INVALID_QUOTES;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.INVALID_TRANSITION_EXTRA_QUOTES;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.INVALID_TRANSITION_QUOTES;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.INVALID_TRANSITION_VARIABLES;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.INVALID_VARIABLES;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.LOOPNODE_INVALID_EXPRESSION;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.TRANSITION_VARIABLE_EMPTY;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.jbpm.gd.jpdl.model.Condition;
import org.jbpm.gd.jpdl.model.Decision;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.ui.model.LoopNode;
import com.tandbergtv.watchpoint.studio.ui.model.WPTransition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.model.ConditionExpression;
import com.tandbergtv.watchpoint.studio.validation.model.DecisionExpression;

public class DecisionExprValidationUtil {
	/* WARNING THIS CLASS IS WRITE-ONLY. IF YOU CAN READ THIS YOU ARE A FUCKING GENIUS */
	/* We don't need to steeenkeeng AST, lexer, stack. No sir. REAL men use regex. */

	private static final String EMPTY_OPERATOR = "empty ";
	
	private static final String [] OPERATORS = { EMPTY_OPERATOR };
	
	private static final String [] KEYWORDS = { "null" };

	private static final String[] REGULAR_EXPRESSION = { 
			"[(][\\s]*[)]",
			"[[a-z][A-Z][0-9]'][\\s]*[=|&][\\s]*[['[A-Z][a-z]][0-9]]", "[|&][\\s]*[')]",
			"[(][\\s]*[|<>&]", "[)][\\s]*[(]", "[=<>][\\s]*[)]", "[)][\\s]*[[a-z][A-Z][0-9]]",
			"[|][\\s]*[&]", "[&][\\s]*[|]", "[|&][\\s]*[=]", "[=][\\s]*[|&]",
			"[|][\\s]*[|][\\s]*[|]", "[&][\\s]*[&][\\s]*[&]", "[=][\\s]*[=][\\s]*[=]",
			"[\\w][\\s]*'[\\s]*[\\w]", "[\\w][\\s]*![\\s]*[\\w]", "[&][!]", "[!][\\s]", "[|][!]",
			"[!][|]", "[&][!]", "[!][&]", "[&][\\s]+[&]", "[(!][)]" 
	};

	private static final char[] ILLEGAL_CHARACTERS = { '+', '-', '*', '/', '?' };

	private static final String PREFIX = "(";

	private static final String SUFFIX = ")";

	private static final String SPACE = " ";

	private static final String EMPTY_STRING = "";

	private static final String SINGLE_QUOTE = "'";

	/**
	 * Validates the Decision Expression
	 *
	 * @param decExprObject
	 * @return valMsg
	 */
	public static Object[] validateDecisonExpr(DecisionExpression decExprObject) {
		Decision decision = decExprObject.getDecision();

        Object[] valMsg = null;
		String expr = decision.getExpression();
		if (expr != null && !"".equals(expr)) {
			valMsg = isDecisionExprValid(expr, decExprObject);
		}
		return valMsg;
	}

	/**
	 * Validates the Condition expression
	 *
	 * @param conditionExprObject
	 * @return val Msg
	 */
	public static Object[] validateConditionExpr(ConditionExpression conditionExprObject) {
		Transition transition = conditionExprObject.getTransition();
		Condition condition = transition.getCondition();

		String expr = (condition != null) ? condition.getExpression() : null;
        Object[] valMsg = null;
		if (expr != null && !"".equals(expr)) {
			valMsg = isConditionExprValid(expr, conditionExprObject);
		}
		return valMsg;
	}

	/**
	 * Gets the operands in the expression along with the indices of their occurence in the
	 * expression.
	 *
	 * @param expression
	 *            the expression from whose operands are required
	 * @return a map of operands and their start indices
	 */
	public static Map<Integer, String> getOperands(String expression) {
		Map<Integer, String> operands = new HashMap<Integer, String>();
		/* FIXME If this is a for-each loop, condition is list[index] != null. Have to replace both variables. */
		char[] specialChars = { '=', '|', '!', '&', '(', ')', '[', ']', '>', '<', '#', '{', '?', ':' };
		int startIndex = 0, index = 0;

		String tempValue = "";
		while (index < expression.length()) {
			char c = expression.charAt(index);
			boolean isValue = true;

			for (int j = 0; j < specialChars.length; j++) {
				if (c == specialChars[j]) {
					isValue = false;
					break;
				}
			}
			
			// Skip past other operators
			if (expression.startsWith(EMPTY_OPERATOR, index)) {
				index += EMPTY_OPERATOR.length();
				startIndex += EMPTY_OPERATOR.length();
				continue;
			}
			
			if (isValue)
				tempValue = tempValue + c;

			if (!tempValue.trim().equals("") && !isValue) {
				for (int i = 0; i < tempValue.length(); i++) {
					if (tempValue.charAt(0) == ' ') {
						startIndex++;
						tempValue = tempValue.substring(1);
					}
				}
				if (tempValue.indexOf('\'') == -1 && isTokenOperand(tempValue.trim())) {
					operands.put(startIndex, tempValue.trim());
				}
				tempValue = "";
			}
			if (!isValue)
				startIndex = index + 1;
			index++;
		}
		return operands;
	}

	/*
	 * checks if the given Condition expression is valid or not
	 *
	 * @return true if expr is valid
	 */
	private static Object[] isConditionExprValid(String expr,
			ConditionExpression conditionExprObject) {
		Transition transition = conditionExprObject.getTransition();
        return isExpressionValid(expr, conditionExprObject.getContainerVariables(), transition);
	}

	/*
	 * checks if the given decison expression is valid or not
	 *
	 * @return true if expr is valid
	 */
	private static Object[] isDecisionExprValid(String expr,
			DecisionExpression decExprObject) {
		Decision decision = decExprObject.getDecision();
		String[] decisionExpr = expr.split("\\?");
		if (decisionExpr.length != 2) {
			return new Object[] {decision, INVALID_DECISION_EXPRESSION};
		}
        Object[] valMsg = isExpressionValid(decisionExpr[0], decExprObject.getContainerVariables(), decision);
		if (valMsg != null) {
			return valMsg;
		}
		if (!isValidTransitionState(decisionExpr[1])) {
			return new Object[] {decision, INVALID_NUMBER_OF_VARIABLES};
		}
		if (!isValidTransitionExpression(decisionExpr[1])) {
			return new Object[] {decision, INVALID_TRANSITION_QUOTES};
		}
		if (!isValidTransExprQuotes(decisionExpr[1])) {
			return new Object[] {decision, INVALID_TRANSITION_EXTRA_QUOTES};
		}
		if (!isTransVariableEmpty(decisionExpr[1])) {
			return new Object[] {decision, TRANSITION_VARIABLE_EMPTY};
		}

		String invalidNames = getInvalidTransitionNames(decisionExpr[1], decExprObject);
		if (invalidNames != null) {
			List<String> parameters = new ArrayList<String>();
			parameters.add(invalidNames);
			return new Object[] {decision, INVALID_TRANSITION_VARIABLES, parameters};
		}
		return null;
	}

    /*
	 * checks if the given expression is valid or not
	 *
	 * @return true if expr is valid
	 */
	public static Object[] isExpressionValid(String expr,
			List<WPVariable> variables, Object obj) {
		expr = expr.trim();
		if (!expr.startsWith(PREFIX) || !expr.endsWith(SUFFIX)) {
			return new Object[] {obj, INVALID_OPEN_CLOSE_PARENTHESIS};
		}
		if (!checkBrackets(expr)) {
			return new Object[] {obj, INVALID_BRACKETS};
		}
		if (!checkQuotes(expr)) {
			return new Object[] {obj, INVALID_QUOTES};
		}
		if (checkExpr(expr)) {
			if (obj instanceof Decision ) {
				return new Object[] {obj, INVALID_DECISION_EXPRESSION};
			} else if (obj instanceof WPTransition) {
				return new Object[] {obj, INVALID_CONDITION_EXPRESSION};
			} else if (obj instanceof LoopNode) {
				return new Object[] {obj, LOOPNODE_INVALID_EXPRESSION};
			}
			return null;
		}
		if (checkIllegalCharacters(expr)) {
			return new Object[] {obj, ValidationMessageCode.ILLEGAL_CHARACTERS};
		}

		String invalidNames = getInvalidVariables(getLogicalOperands(expr), variables);
		if (invalidNames != null) {
			List<String> parameters = new ArrayList<String>();
			parameters.add(invalidNames);
			return new Object[] {obj, INVALID_VARIABLES, parameters};
		}
		return null;
	}

	/*
	 * checks if there is any missing brackets
	 *
	 * @return true if expr is valid
	 */
	private static boolean checkBrackets(String expr) {
		int sLength = expr.length();
		int inBracket = 0;

		for (int i = 0; i < sLength; i++) {
			if (expr.charAt(i) == '(' && inBracket >= 0)
				inBracket++;
			else if (expr.charAt(i) == ')')
				inBracket--;
		}
		if (inBracket != 0)
			return false;
		return true;
	}

	/*
	 * checks if there is any missing quotes
	 *
	 * @return true if expr is valid
	 */
	private static boolean checkQuotes(String expr) {
		int sLength = expr.length();
		int inQuote = 0;

		for (int i = 0; i < sLength; i++) {
			if (expr.charAt(i) == '\'')
				inQuote++;

		}
		if (!(inQuote % 2 == 0))
			return false;
		return true;
	}

	/*
	 * Checks whether the expression is valid or not
	 * Returns true if invalid false otherwise	
	 */
	private static boolean checkExpr(String expr) {
		for (String regExpr : REGULAR_EXPRESSION) {
			Pattern pattern = Pattern.compile(regExpr);
			Matcher matcher = pattern.matcher(expr);
			if (matcher.find()) {
				return true;
			}
		}
		return false;
	}

	/*
	 * Gets the operands in the expression
	 */
	private static String[] getLogicalOperands(String expression) {
		/* FIXME WHY THE FUCK IS getOperands() AN EXACT COPY??? */
		char[] operators = { '=', '|', '!', '&', '(', ')', '>', '<', '[', ']' };
		int index = 0;
		int exprLength = expression.length();
		String tempValue = EMPTY_STRING;
		StringBuffer operandValue = new StringBuffer();
		List<String> operands = new ArrayList<String>();

		while (index < exprLength) {
			char c = expression.charAt(index);
			boolean isValue = true;

			for (int j = 0; j < operators.length; j++) {
				if (c == operators[j]) {
					isValue = false;
					break;
				}
			}

			// Skip past other operators
			if (expression.startsWith(EMPTY_OPERATOR, index)) {
				index += EMPTY_OPERATOR.length();
				continue;
			}

			if (isValue) {
				operandValue.append(expression.charAt(index));
				tempValue = operandValue.toString().replaceAll(SPACE, EMPTY_STRING);
			}

			if ((!tempValue.equals(EMPTY_STRING) && !isValue) || (index == exprLength - 1 && isValue)) {
				if (tempValue.indexOf(SINGLE_QUOTE) == -1 && !isConstant(tempValue))
					operands.add(operandValue.toString().trim());

				operandValue.delete(0, operandValue.length());
				tempValue = EMPTY_STRING;
			}

			index++;
		}

		return operands.toArray(new String[operands.size()]);
	}

	/*
	 * Checks illegal characters in the operands
	 */
	private static boolean checkIllegalCharacters(String expr) {
		boolean illegal = false;
		String[] operands = getLogicalOperands(expr);

		if (operands != null && operands.length > 0) {
			StringBuffer tempString = new StringBuffer();

			for (String operand : operands)
				tempString.append(operand.trim());

			String operandsString = tempString.toString();

			for (int j = 0; j < ILLEGAL_CHARACTERS.length; j++) {
				int index = operandsString.indexOf(ILLEGAL_CHARACTERS[j]);

				if (index != -1) {
					illegal = true;
					break;
				}
			}
		}

		return illegal;
	}

	/*
	 * Gets the list of invalid variables used in the expression, or null if none exist.
	 */
	private static String getInvalidVariables(String[] exprVariables,
			List<WPVariable> variables) {
		List<String> validExceptions = Arrays.asList("null");
		Set<String> invalidNames = new HashSet<String>();
		boolean isValidVariable = false;
		for (String exprVariable : exprVariables) {
			if(validExceptions.contains(exprVariable)){
				continue;
			}
			
			isValidVariable = false;
			for (WPVariable variable : variables) {
				if (exprVariable.trim().equals(variable.getName())) {
					isValidVariable = true;
					break;
				}
			}
			if (!isValidVariable) {
				invalidNames.add(exprVariable.trim());
			}
		}

		return buildCSVString(invalidNames);
	}

	/*
	 * Checks whether the decision is transitioned to a valid state
	 */
	private static boolean isValidTransitionState(String expr) {
		String[] transitionStates = expr.split(":");
		if (transitionStates.length != 2) {
			return false;
		}
		return true;
	}

	/*
	 * Gets the list of transition names in the expression that are invalid or null if none exist.
	 */
	private static String getInvalidTransitionNames(String expr, DecisionExpression exprObject) {
		Set<String> invalidNames = new HashSet<String>();
		boolean isValid = false;
		List<Transition> transitions = getTransitionVariables(exprObject.getDecision());
		String[] transitionStates = expr.split(":");
		
		for (String transitionState : transitionStates) {
			transitionState = transitionState.trim().replaceAll(SINGLE_QUOTE, EMPTY_STRING);	
			isValid = false;
			for (Transition transition : transitions) {
				if (transitionState.equals(transition.getName())) {
					isValid = true;
					break;
				}
			}
			if (!isValid) {
				invalidNames.add(transitionState.trim());
			}
		}

		return buildCSVString(invalidNames);
	}

	/* Builds a single comma separated value string from the values in the list. */
	private static String buildCSVString(Collection<String> values) {
		StringBuilder buf = new StringBuilder();
		boolean first = true;
		for (String name : values) {
			buf.append(first ? "" : ", ");
			first = false;
			buf.append(name);
		}

		return (buf.length() == 0) ? null : buf.toString();
	}

	/*
	 * This method retrieves the defined Transition states in Decison Node
	 *
	 * @return Array of Strings
	 */
	private static List<Transition> getTransitionVariables(Decision decision) {
		List<Transition> variables = new ArrayList<Transition>();
		Transition[] transitionVariables = decision.getTransitions();
		for (Transition variable : transitionVariables) {
			variables.add((Transition) variable);
		}
		return variables;
	}

	/*
	 * Counts if there are any missing/extra quotes
	 *
	 * @return true if expr is valid
	 */
	private static int checkQuoteCount(String expr) {
		int sLength = expr.length();
		int inQuote = 0;

		for (int i = 0; i < sLength; i++) {
			if (expr.charAt(i) == '\'')
				inQuote++;
		}
		return inQuote;
	}

	/*
	 * checks whether the transition expression is correct or not
	 *
	 * @return true if expr is valid
	 */
	private static boolean isValidTransitionExpression(String expr) {
		String[] transVariables = expr.split(":");
		for (String transVariable : transVariables) {
			if (!transVariable.trim().startsWith("'") || !transVariable.trim().endsWith("'")) {
				return false;
			}

		}
		return true;
	}

	/*
	 * checks whether the transition variable is empty
	 *
	 * @return true if expr is valid
	 */
	private static boolean isTransVariableEmpty(String expr) {
		String[] transVariables = expr.split(":");
		for (String transVariable : transVariables) {
			if (transVariable.equals("''")) {
				return false;
			}

		}
		return true;
	}

	/*
	 * checks whether the quotes for transition variable is correct or not
	 *
	 * @return true if expr is valid
	 */
	private static boolean isValidTransExprQuotes(String expr) {
		String[] transVariables = expr.split(":");
		for (String transVariable : transVariables) {
			if (checkQuoteCount(transVariable) != 2) {
				return false;
			}
		}
		return true;
	}
	
	private static boolean isTokenOperand(String s) {
		 return (!(isOperator(s) || isKeyWord(s) || isConstant(s)));
	}
	
	private static boolean isOperator(String s) {
		for (String op : OPERATORS)
			if (op.equals(s))
				return true;
		return false;
	}
	
	private static boolean isKeyWord(String s) {
		for (String keyword : KEYWORDS)
			if (keyword.equals(s))
				return true;
		return false;
	}

	/*
	 * Checks whether the operand is constant or not
	 */
	private static boolean isConstant(String operand) {
		try {
			new BigDecimal(operand);
			return true;
		} catch (NumberFormatException e) {
		}
		
		return false;
	}
}
