/*
 * Created on Dec 11, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.nodeelement;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Validation Rule applied to any Node to ensure that all the outgoing transitions from the Node do
 * not have the same destination (the transitions cannot be parallel).
 * 
 * @author Vijay Silva
 */
public class NodeTransitionsParallelRule extends ValidationRule<NodeElement>
{
	/**
	 * Validate that none of the outgoing transitions from the Node Element are parallel to each
	 * other. Adds a validation rule for each common target node in the transition set.
	 * 
	 * @param nodeElement
	 *            The Node Element to validate
	 * 
	 * @return The list of Validation Messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(NodeElement nodeElement)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		Map<String, Integer> targets = new HashMap<String, Integer>();

		Transition[] transitions = nodeElement.getTransitions();
		if (transitions != null)
		{
			for (Transition transition : transitions)
			{
				String target = transition.getTo();
				Integer value = targets.get(target);
				int count = (value != null) ? value.intValue() + 1 : 1;
				targets.put(target, count);
			}
		}

		/* validate the parallel transitions */
		this.validateParallelTransitions(nodeElement, targets, messages);

		return messages;
	}

	/*
	 * Validates for parallel transitions by generating a validation message for each of the
	 * duplicate targets present.
	 */
	private void validateParallelTransitions(NodeElement nodeElement, Map<String, Integer> targets,
			List<ValidationMessage> messages)
	{
		ValidationMessageCode code = ValidationMessageCode.NODE_TRANSITIONS_PARALLEL;

		for (String target : targets.keySet())
		{
			Integer count = targets.get(target);
			if (count.intValue() > 1)
			{
				List<String> parameters = new ArrayList<String>();
				parameters.add(target);
				parameters.add(count.toString());
				ValidationMessageAdder.getInstance().addValidationMessage(messages, nodeElement, code, parameters);
			}
		}
	}
}
