/**
 * ExecutionContext.java
 * Created Sep 21, 2010
 */
package com.tandbergtv.watchpoint.studio.debugger.core.graph.exe;

import static com.tandbergtv.watchpoint.studio.debugger.core.graph.exe.ListVariableInstance2.END_TOKEN;
import static com.tandbergtv.watchpoint.studio.debugger.core.graph.exe.ListVariableInstance2.START_TOKEN;

import java.util.Map;

import org.jbpm.JbpmException;
import org.jbpm.context.exe.ContextInstance;
import org.jbpm.context.exe.TokenVariableMap;
import org.jbpm.context.exe.VariableInstance;
import org.jbpm.context.exe.variableinstance.ByteArrayInstance;
import org.jbpm.context.exe.variableinstance.UnpersistableInstance;
import org.jbpm.graph.exe.Token;

import bsh.Interpreter;

/**
 * A subclass of JBPM's {@link org.jbpm.graph.exe.ExecutionContext} for performing additional 
 * validation etc
 * 
 * @author Sahil Verma
 */
public class ExecutionContext2 extends org.jbpm.graph.exe.ExecutionContext {

	/**
	 * Creates an ExecutionContext
	 * 
	 * @param token
	 */
	public ExecutionContext2(Token token) {
		super(token);
	}

	/**
	 * @param other
	 */
	public ExecutionContext2(org.jbpm.graph.exe.ExecutionContext other) {
		super(other);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object getVariable(String name) {
		return evaluate(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setVariable(String name, Object value) {
		if (name.length() == 0)
			throw new JbpmException("Variable name is an empty string");
		
		if (value != null) {
			VariableInstance variable = VariableInstance.createVariableInstance(value);

			if (variable instanceof UnpersistableInstance || variable instanceof ByteArrayInstance)
				throw new JbpmException("Variable " + name + " uses unsupported data type - " + value.getClass().getName());
			
			if (value.toString().length() >= 4000)
				throw new JbpmException("Variable " + name + " exceeds max length (4000)");
		}
		
		if (ListVariableInstance2.check(name)) {
			ListVariableInstance2 list = getList(name);
			int index = index(name);
			
			list.set(index, value);
		} else {
			super.setVariable(name, value);
		}
	}
	
	/**
	 * Gets a list variable instance using the expression which must be in the format name[index]
	 * 
	 * @param name
	 * @return
	 */
	private ListVariableInstance2 getList(String name) {
		String variableName = ListVariableInstance2.parseVariableName(name);
		VariableInstance v = getVariableInstance(variableName);
		
		if (!(v instanceof ListVariableInstance2))
			throw new IllegalArgumentException(token + ", " + variableName + " is not a list");
		
		return (ListVariableInstance2) v;
	}
	
	private VariableInstance getVariableInstance(String name) {
		ContextInstance ci = token.getProcessInstance().getContextInstance();
		TokenVariableMap map = ci.getTokenVariableMap(token);
		
		if (map == null)
			return null;
		
		return map.getVariableInstance(name);
	}

	private Integer index(String name) {
		int start = name.indexOf(START_TOKEN);
		int end = name.indexOf(END_TOKEN);
		String s = name.substring(start + 1, end);
		Object value = token.getProcessInstance().getContextInstance().getVariable(s);
		
		if (value instanceof Integer)
			return (Integer) value;
		
		return Integer.valueOf(s);
	}
	
	/**
	 * 		Evaluates a variable value using BeanShell. Variable values may be represented as scripts.
	 * 
	 * @param expression
	 * @return
	 */
	protected Object evaluate(String expression) {
		if (expression == null || expression.trim().isEmpty())
			throw new JbpmException("Missing variable name or expression");

		Map<?, ?> variables = getContextInstance().getVariables(token);
		
		// checks if its a single variable
		if (variables.containsKey(expression))
			return super.getVariable(expression);

		try {
			Interpreter i = new Interpreter(); // Construct an interpreter

			// put all the node variables in the interpreter context.
			for (Object key : variables.keySet())
				i.set((String) key, variables.get(key));

			/* Add special variables */
			i.set("token", token);
			i.set("processInstance", token.getProcessInstance());
			i.set("_rootdir", System.getProperty("com.tandbergtv.cms.product.dir"));
			
			// FIXME Load script variable providers and import packages etc
			return i.eval(expression);
		} catch (Exception error) {
			throw new JbpmException("Error processing expression '" + expression + "'", error);
		}
	}
}
