/*******************************************************************************
 * Copyright (c) 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Bjorn Freeman-Benson - initial API and implementation
 *******************************************************************************/
package com.tandbergtv.watchpoint.studio.debugger.model;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.PlatformObject;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IDebugElement;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.internal.ui.viewers.model.provisional.IElementLabelProvider;

import com.tandbergtv.watchpoint.studio.debugger.WatchpointTemplateDebuggerPlugin;
import com.tandbergtv.watchpoint.studio.debugger.launching.TemplateLauncherConstants;
import com.tandbergtv.watchpoint.studio.debugger.ui.JpdlVariableLabelProvider;

/**
 * Common function of Jpdl debug model elements
 */
public abstract class JpdlDebugElement extends PlatformObject implements IDebugElement {

    // containing target
    protected JpdlDebugTarget fTarget;

    /**
     * Constructs a new debug element contained in the given debug target.
     * 
     * @param target
     *            debug target (Jpdl Wrapper)
     */
    public JpdlDebugElement(JpdlDebugTarget target) {
        fTarget = target;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IDebugElement#getModelIdentifier()
     */
    public String getModelIdentifier() {
        return TemplateLauncherConstants.ID_JPDL_DEBUG_MODEL;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IDebugElement#getDebugTarget()
     */
    public IDebugTarget getDebugTarget() {
        return fTarget;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.debug.core.model.IDebugElement#getLaunch()
     */
    public ILaunch getLaunch() {
        return getDebugTarget().getLaunch();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
     */
    public Object getAdapter(Class adapter) {
        if (adapter == IDebugElement.class) {
            return this;
        } else if (adapter == IElementLabelProvider.class) {
            return new JpdlVariableLabelProvider();
        }
        return super.getAdapter(adapter);
    }

    protected void abort(String message, Throwable e) throws DebugException {
        throw new DebugException(new Status(IStatus.ERROR, WatchpointTemplateDebuggerPlugin.getDefault()
                .getDescriptor().getUniqueIdentifier(), DebugPlugin.INTERNAL_ERROR, message, e));
    }

    /**
     * Fires a debug event
     * 
     * @param event
     *            the event to be fired
     */
    protected void fireEvent(DebugEvent event) {
        DebugPlugin.getDefault().fireDebugEventSet(new DebugEvent[] { event });
    }

    /**
     * Fires a <code>CREATE</code> event for this element.
     */
    protected void fireCreationEvent() {
        fireEvent(new DebugEvent(this, DebugEvent.CREATE));
    }

    /**
     * Fires a <code>RESUME</code> event for this element with the given detail.
     * 
     * @param detail
     *            event detail code
     */
    public void fireResumeEvent(int detail) {
        fireEvent(new DebugEvent(this, DebugEvent.RESUME, detail));
    }

    /**
     * Fires a <code>SUSPEND</code> event for this element with the given
     * detail.
     * 
     * @param detail
     *            event detail code
     */
    public void fireSuspendEvent(int detail) {
        fireEvent(new DebugEvent(this, DebugEvent.SUSPEND, detail));
    }

    /**
     * Fires a <code>TERMINATE</code> event for this element.
     */
    protected void fireTerminateEvent() {
        fireEvent(new DebugEvent(this, DebugEvent.TERMINATE));
    }
}
