/**
 * ListVariableInstance.java
 * Created Nov 3, 2010
 */
package com.tandbergtv.watchpoint.studio.debugger.core.graph.exe;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import org.jbpm.context.exe.ContextInstance;
import org.jbpm.context.exe.TokenVariableMap;
import org.jbpm.context.exe.VariableInstance;

/**
 * A list variable instance. Stores a placeholder value and adds all list elements as variables.
 * 
 * @author Sahil Verma
 */
public class ListVariableInstance2 extends VariableInstance {

	private static final long serialVersionUID = 5567529337015616053L;
	
	protected final String value = "LIST";
	
	public static final String START_TOKEN = "[";
	
	public static final String END_TOKEN = "]";
	
	/**
	 * The list item separator
	 */
	public static final String SEPARATOR = ",";
	
	public ListVariableInstance2() {
		super();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isStorable(Object value) {
		return value instanceof List<?>;
	}
	
	/**
	 * If no [ or ], not a list item. Otherwise ensures that the format is name[index], where index
	 * is a positive integer.
	 * 
	 * @param name the expression
	 * @return
	 */
	public static boolean check(String name) {
		if (!name.contains(START_TOKEN) && !name.contains(END_TOKEN))
			return false;
		
		if (name.contains(START_TOKEN) && !name.endsWith(END_TOKEN)) //No I dont know regex
			throw new IllegalArgumentException("Invalid expression: " + name);	
		
		if (name.endsWith(END_TOKEN) && !name.contains(START_TOKEN))
			throw new IllegalArgumentException("Invalid expression: " + name);
		
		return true;
	}
	
	/**
	 * Returns the name of the list variable, input string must be in the format name[index].
	 * 
	 * @param name the expression
	 * @return
	 */
	public static String parseVariableName(String name) {
		if (!check(name))
			throw new IllegalArgumentException("Invalid list expression: " + name);
		
		return name.substring(0, name.indexOf(START_TOKEN));
	}

	/**
	 * Returns the item at the specified index
	 * 
	 * @param index
	 * @return
	 */
	public Object get(int index) {
		rangeCheck(index);
		
		return ((List<?>) getObject()).get(index);
	}
	
	/**
	 * Sets the item at the specified index
	 * 
	 * @param index
	 * @param value
	 */
	public void set(int index, Object value) {
		rangeCheck(index);
		
		@SuppressWarnings("unchecked")
		List<String> list = (List<String>) getObject(); 
		
		list.set(index, value.toString());
		
		setObject(list);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	protected Object getObject() {
		ContextInstance ci = token.getProcessInstance().getContextInstance();
		TokenVariableMap map = ci.getTokenVariableMap(token);
		SortedMap<Integer, String> sorted = new TreeMap<Integer, String>();
		
		for (VariableInstance variable : (Collection<VariableInstance>) map.getVariableInstances().values()) {
			if (variable.getName().startsWith(name + START_TOKEN)) {
				Integer index = index(variable.getName());
				sorted.put(index, variable.getValue().toString());
			}
		}
		
		return new ArrayList<String>(sorted.values());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void setObject(Object value) {
		ContextInstance ci = token.getProcessInstance().getContextInstance();
		
		removeListVariables(ci);
		
		List<?> list = (List<?>) value;
		
		for (int i = 0; i < list.size(); i++)
			ci.setVariable(name + START_TOKEN + i + END_TOKEN, list.get(i), token);
		
		/* Reset these variables so that we don't ever use stale values */
		valueCache = null;
		isValueCached = false;
	}
	
	private void removeListVariables(ContextInstance ci) {
		TokenVariableMap map = ci.getTokenVariableMap(token);
		
		if (map != null) {
			@SuppressWarnings("unchecked")
			Map<String, VariableInstance> variables = map.getVariableInstances();
			
			if (variables != null) {
				for (Iterator<VariableInstance> i = variables.values().iterator(); i.hasNext(); ) {
					VariableInstance variable = i.next();
					
					if (variable.getName().startsWith(name + START_TOKEN))
						i.remove();
				}
			}
		}
	}
	
	private Integer index(String name) {
		int start = name.indexOf(START_TOKEN);
		int end = name.indexOf(END_TOKEN);
		String s = name.substring(start + 1, end);
		Object value = token.getProcessInstance().getContextInstance().getVariable(s);
		
		if (value instanceof Integer)
			return (Integer) value;
		
		return Integer.valueOf(s);
	}
	
	private void rangeCheck(int index) {
		List<?> list = (List<?>) getObject();
		
		if (index < 0 || index >= list.size())
			throw new IndexOutOfBoundsException("Invalid index " + index + " for " + name + ", size = " + list.size());
	}
}
