/*
 * Created on Jun 22, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dataaccess.jpa;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.EntityTransaction;
import javax.persistence.LockModeType;

import com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext;

/**
 * The Persistence Context for data access using JPA. This implementation is not thread-safe.
 * 
 * @author Vijay Silva
 */
public class JPAPersistenceContext implements IPersistenceContext
{
	private EntityManagerFactory factory;

	private EntityManager entityManager;

	private LockModeType lockMode = LockModeType.READ;

	/**
	 * Constructor
	 * 
	 * @param entityManagerFactory
	 *            Entity Manager Factory that is used to create a Entity Manager for data access.
	 */
	public JPAPersistenceContext(EntityManagerFactory entityManagerFactory)
	{
		this.factory = entityManagerFactory;
	}

	/**
	 * @return the entityManager
	 */
	public EntityManager getEntityManager()
	{
		return this.entityManager;
	}

	/**
	 * @return the lockMode to use when locking entities
	 */
	public LockModeType getLockMode()
	{
		return this.lockMode;
	}

	/**
	 * @param lockMode
	 *            the lockMode to set when locking entities
	 */
	public void setLockMode(LockModeType lockMode)
	{
		if (lockMode == null)
			lockMode = LockModeType.READ;

		this.lockMode = lockMode;
	}

	/**
	 * Creates a new Entity Manager in a thread-safe manner
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext#initialize()
	 */
	public void initialize()
	{
		// Check if the context is already initialized
		if (this.isInitialized())
			return;

		this.entityManager = this.factory.createEntityManager();
	}

	/**
	 * Close the Entity Manager in a thread-safe manner
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext#close()
	 */
	public void close()
	{
		if (!this.isInitialized())
			return;

		this.entityManager.close();
		this.entityManager = null;
	}

	/**
	 * Check to see if this context has already been initialized
	 * 
	 * @return true if the context is initialized, false otherwise
	 */
	protected boolean isInitialized()
	{
		return (this.entityManager != null && this.entityManager.isOpen());
	}

	/**
	 * Begins a new Transaction. Throws an exception if the Transaction has already been started.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext#beginTransaction()
	 */
	public void beginTransaction()
	{
		this.entityManager.getTransaction().begin();
	}

	/**
	 * Commits the current Transaction. Throws a runtime exception if the Transaction has not been
	 * started.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext#commitTransaction()
	 */
	public void commitTransaction()
	{
		this.entityManager.getTransaction().commit();
	}

	/**
	 * Rollback the current Transaction. Throws a runtime exception if the Transaction has not been
	 * started.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext#rollbackTransaction()
	 */
	public void rollbackTransaction()
	{
		this.entityManager.getTransaction().rollback();
	}

	/**
	 * Returns the current Transaction associated with the context.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext#getCurrentTransaction()
	 */
	public EntityTransaction getCurrentTransaction()
	{
		return this.entityManager.getTransaction();
	}
}
