package com.tandbergtv.watchpoint.studio.external.fs.dao;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.HashSet;
import java.util.Set;

import javax.xml.transform.TransformerException;
import javax.xml.xpath.XPathException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.ui.IFileEditorInput;

import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.ui.editor.WatchPointTemplateEditor;
import com.tandbergtv.watchpoint.studio.ui.model.ProtectionKey;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.util.FileUtil;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

public class TemplateParser {

	/**
	 * Parses a template
	 * 
	 * @param file
	 * @return
	 * @throws TransformerException 
	 * @throws XPathException 
	 * @throws IOException 
	 */
	public static WorkflowTemplateDTO parse(File file) throws Exception {
		String xml = new String(FileUtil.readFile(file.getAbsolutePath()));
		return parse(xml);
	}
	
	/**
	 * Parses a template
	 * 
	 * @param file
	 * @return
	 * @throws TransformerException 
	 * @throws XPathException 
	 * @throws CoreException 
	 * @throws IOException 
	 */
	public static WorkflowTemplateDTO parse(IFile file) throws Exception {
		String xml = new String(FileUtil.readFile(file.getContents()));
		WorkflowTemplateDTO template = parse(xml);
		template.setPath(file.getFullPath().toPortableString());
		template.setProjectName(file.getProject().getName());
		return template;
	}

	/**
	 * Parses a template
	 * 
	 * @param file
	 * @return
	 * @throws TransformerException 
	 * @throws XPathException 
	 * @throws CoreException 
	 * @throws IOException 
	 */
	public static WorkflowTemplateDTO parse(WatchPointTemplateEditor editor) throws Exception {
		IFileEditorInput in = (IFileEditorInput) editor.getEditorInput();
		
		String xml = new String(FileUtil.readFile(in.getFile().getContents()));
		
		WorkflowTemplateDTO template = parse(xml);
		template.setPath(in.getFile().getFullPath().toPortableString());
		template.setProjectName(in.getFile().getProject().getName());
		
		return template;
	}


	/**
	 * Parses a template
	 * 
	 * @param file
	 * @return
	 * @throws TransformerException 
	 * @throws XPathException 
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static WorkflowTemplateDTO parse(String xml) throws Exception {
		WorkflowTemplateDTO template = new WorkflowTemplateDTO();

		WorkflowTemplate model = SemanticElementUtil.createWorkflowTemplate(xml);

		parseProperties(template, model);
		parseNodeElements(template, model);

		return template;
	}
	
	/**
	 * 		Parses the template's NodeElements (NodeDefinitions).
	 * 	
	 * @param template
	 * @param model
	 */
	private static void parseNodeElements(WorkflowTemplateDTO template, WorkflowTemplate model) {
		ExtractDTOElementsVisitor visitor = new ExtractDTOElementsVisitor(template);
		model.accept(visitor);
        template.setMessages(visitor.getMessages());
        template.setSubprocesses(visitor.getSubProcesses());
        template.setManualTaskNodes(visitor.getManTaskNodes());

		Set<com.tandbergtv.watchpoint.studio.dto.ProtectionKey> protectionKeys = new HashSet<com.tandbergtv.watchpoint.studio.dto.ProtectionKey>();
		for (ProtectionKey uiPk : model.getProtectionKeys()) {
			com.tandbergtv.watchpoint.studio.dto.ProtectionKey protectionKey = new com.tandbergtv.watchpoint.studio.dto.ProtectionKey();
			protectionKey.setName(uiPk.getName());
			protectionKeys.add(protectionKey);
		}
		
		template.setProtectionKeys(protectionKeys);
	}
	
	/**
	 * 		Parses the template properties. 
	 * 
	 * @param template
	 * @param model
	 */
	private static void parseProperties(WorkflowTemplateDTO template, WorkflowTemplate model) {
		String name = model.getName();

		template.setName(name);

		template.setVersion(model.getVersion());
	}

}
