/*
 * Created on Mar 28, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.external.jpf.validation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.java.plugin.registry.Extension;
import org.java.plugin.registry.IntegrityCheckReport;
import org.java.plugin.registry.Extension.Parameter;
import org.java.plugin.registry.IntegrityCheckReport.ReportItem;
import org.java.plugin.registry.IntegrityCheckReport.Severity;

import com.tandbergtv.watchpoint.studio.external.jpf.PluginParameters;

/**
 * Utility class that performs validation of the Resource Type Plugin Extension.
 * 
 * @author Vijay Silva
 */
public final class ResourceTypeExtensionValidator
{
	/*
	 * Constructor
	 */
	private ResourceTypeExtensionValidator()
	{ // Cannot Instantiate
	}

	/**
	 * Validates the Extension using JPF framework validation and some additional validation to make
	 * sure all required fields are present.
	 * 
	 * @param extension
	 *            The Extension to validate
	 * @param report
	 *            The IntegrityCheckReport generated by the framework
	 * 
	 * @return The list of error messages, or empty list if valid.
	 */
	public static List<String> validate(Extension extension, IntegrityCheckReport report)
	{
		List<String> result = new ArrayList<String>();

		validateExtension(extension, report, result);

		if (result.size() == 0)
		{
			validateCommunication(extension, result);
		}

		return result;
	}

	/*
	 * Validates the Extension using JPF validation
	 */
	private static void validateExtension(Extension extension, IntegrityCheckReport report,
			List<String> result)
	{
		if (!extension.isValid())
		{
			result.add("The Plugin Extension has failed framework validation.");

			// Get the list of validation errors
			for (ReportItem item : report.getItems())
			{
				if (!(item.getSeverity() == Severity.ERROR))
					continue;

				String sourceId = (item.getSource() != null) ? item.getSource().getId() : null;
				String pluginId = extension.getDeclaringPluginDescriptor().getId();
				String extensionId = extension.getId();
				if (extensionId.equals(sourceId) || pluginId.equals(sourceId))
				{
					result.add(item.getMessage());
				}
			}
		}
	}

	/*
	 * Validate the Communication Parameter in the Extension
	 */
	private static void validateCommunication(Extension extension, List<String> result)
	{
		Parameter param = extension.getParameter(PluginParameters.COMMUNICATION);
		if (param != null)
		{
			validateAdaptor(param.getSubParameter(PluginParameters.ADAPTOR), result);
			//messages are not required anymore
			if( param.getSubParameter(PluginParameters.MESSAGES) != null ){
				validateMessageUIDs(param.getSubParameter(PluginParameters.MESSAGES), result);
			}
		}
	}

	/*
	 * Validate the Adaptor properties given the Adaptor parameter
	 */
	private static void validateAdaptor(Parameter parameter, List<String> result)
	{
		// Check the Adaptor custom and default parameters
		Parameter customParam = parameter.getSubParameter(PluginParameters.ADAPTOR_CUSTOM);
		Parameter defaultParam = parameter.getSubParameter(PluginParameters.ADAPTOR_DEFAULT);
		boolean isCustom = (customParam != null);
		boolean isDefault = (defaultParam != null);

		if (isCustom == isDefault)
		{ // This is an error condition, must be either custom or default
			if (isCustom)
			{
				String msg = "The Adaptor configuration cannot contain both 'default' and "
						+ "'custom' parameters.";
				result.add(msg);
			}
			else
			{
				String msg = "The Adaptor configuration must contain one of 'default' or "
						+ "'custom' parameters.";
				result.add(msg);
			}
		}
	}

	/*
	 * Validate the Message UIDs (for uniqueness) given the Messages parameter
	 */
	private static void validateMessageUIDs(Parameter parameter, List<String> result)
	{
		// Ensure that all the Message UID values are unique.
		Collection<Parameter> messageParams = parameter.getSubParameters(PluginParameters.MESSAGE);
		List<String> uidValues = new ArrayList<String>();
		List<String> duplicates = new ArrayList<String>();

		for (Parameter messageParam : messageParams)
		{
			Parameter uidParam = messageParam.getSubParameter(PluginParameters.MESSAGE_UID);
			String uid = uidParam.valueAsString();
			if (uidValues.contains(uid) && !duplicates.contains(uid))
				duplicates.add(uid);

			uidValues.add(uid);
		}

		for (String uid : duplicates)
		{
			result.add("The Message UID: " + uid + " is not uniquely assigned to a message.");
		}
	}
}
