/*
 * Created on Sep 24, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.widgets.FormToolkit;

import com.tandbergtv.watchpoint.studio.dto.AdaptorType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.eventhandler.FormPartDirtyModifyListener;

/**
 * The Section Part containing the Adaptor Type
 * 
 * @author Vijay Silva
 */
public class AdaptorTypeSectionPart extends ResourceTypeSectionPart
{
	Label typeLabel;

	CCombo typeCombo;

	private List<AdaptorType> adaptorTypes = new ArrayList<AdaptorType>();

	// ========================================================================
	// ===================== CONSTRUCTORS
	// ========================================================================

	/**
	 * @param parentPart
	 *            The Part that contains this composite
	 * @param parent
	 *            The Parent Composite containing the section to be created
	 * @param toolkit
	 *            The toolkit to create the section with
	 * @param style
	 *            The style bits to create the section with
	 */
	public AdaptorTypeSectionPart(IEditorPart parentPart, Composite parent, FormToolkit toolkit,
			int style)
	{
		super(parentPart, parent, toolkit, style);
	}

	// ========================================================================
	// ===================== ACCESSORS
	// ========================================================================

	/**
	 * Gets the Adaptor Type current displayed in the Type ComboBox
	 * 
	 * @return The Adaptor Type (or AdaptorType.NONE if no selection is present)
	 */
	public AdaptorType getAdaptorType()
	{
		int index = this.typeCombo.getSelectionIndex();
		AdaptorType adaptorType = AdaptorType.NONE;
		if (index != -1)
			adaptorType = this.adaptorTypes.get(index);

		return adaptorType;
	}

	/**
	 * Get the Type ComboBox displayed in this section part
	 * 
	 * @return The Type ComboBox
	 */
	public CCombo getTypeCombo()
	{
		return this.typeCombo;
	}

	// ========================================================================
	// ===================== OVERRIDDEN SECTION PART METHODS
	// ========================================================================

	/**
	 * @see org.eclipse.ui.forms.AbstractFormPart#commit(boolean)
	 */
	@Override
	public void commit(boolean onSave)
	{
		super.commit(onSave);

		/* Update the Resource Type with the data in the widgets */
		ResourceType resourceType = this.getResourceType();
		if (resourceType != null)
		{
			int index = this.typeCombo.getSelectionIndex();
			AdaptorType adaptorType = AdaptorType.NONE;
			if (index != -1)
				adaptorType = this.adaptorTypes.get(index);
			resourceType.setAdaptorType(adaptorType);
		}
	}

	// ========================================================================
	// ===================== USER CONTROL INITIALIZATION
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#createSectionClientControls(org.eclipse.ui.forms.IManagedForm)
	 */
	@Override
	protected void createSectionClientControls(IManagedForm managedForm)
	{
		FormToolkit toolkit = managedForm.getToolkit();

		/* The Adaptor Type Label and Combo */
		this.typeLabel = toolkit.createLabel(this.sectionClient, "Adaptor Type: ");
		int style = toolkit.getBorderStyle() | SWT.READ_ONLY;
		this.typeCombo = new CCombo(this.sectionClient, style);
		toolkit.adapt(this.typeCombo, true, true);
		this.typeCombo.addModifyListener(new FormPartDirtyModifyListener(this));
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#initializeSectionClientLayout(org.eclipse.ui.forms.IManagedForm)
	 */
	@Override
	protected void initializeSectionClientLayout(IManagedForm managedForm)
	{
		/* Set the Layout for the Section Client in the Form */
		this.sectionClient.setLayout(this.createGridLayout(2, false));

		/* Set the default Layout data for the widgets */
		this.typeLabel.setLayoutData(this.createLabelGridData());
		this.typeCombo.setLayoutData(this.createGridData());
	}

	// ========================================================================
	// ===================== DISPLAYING THE INPUT DATA
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#populateSectionData()
	 */
	@Override
	protected void populateSectionData()
	{
		ResourceType resourceType = this.getResourceType();

		this.typeCombo.removeAll();
		this.adaptorTypes.clear();
		for (AdaptorType adaptorType : AdaptorType.values())
		{
			this.adaptorTypes.add(adaptorType);
			this.typeCombo.add(getAdaptorTypeDisplayName(adaptorType));
		}

		int defaultIndex = this.adaptorTypes.indexOf(AdaptorType.NONE);
		int index = defaultIndex;
		if (resourceType != null)
		{
			index = this.adaptorTypes.indexOf(resourceType.getAdaptorType());
			if (index == -1)
				index = defaultIndex;
		}

		this.typeCombo.select(index);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#setWidgetPermissions()
	 */
	@Override
	protected void setWidgetPermissions()
	{
		boolean isEditable = this.isInputEditable();
		this.typeCombo.setEnabled(isEditable);
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/* Get a display string for the Adaptor Type */
	private String getAdaptorTypeDisplayName(AdaptorType adaptorType)
	{
		String value = adaptorType.toString();

		switch (adaptorType)
		{
			case NONE:
				value = "None";
				break;

			case CUSTOM:
				value = "Custom";
				break;

			case DEFAULT:
				value = "Default";
				break;
		}

		return value;
	}
}
