package com.tandbergtv.watchpoint.studio.ui.properties;

import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.CheckboxCellEditor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.ComboBoxViewerCellEditor;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.ui.editor.expression.ExpressionEditor;
import com.tandbergtv.watchpoint.studio.ui.model.IExpression;
import com.tandbergtv.watchpoint.studio.ui.model.IValidatableElement;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPAbstractNode;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.properties.nodedefinition.CustomComboBoxCellEditor;
import com.tandbergtv.watchpoint.studio.ui.properties.nodedefinition.NodeDefinitionVariablesCellModifier;
import com.tandbergtv.watchpoint.studio.ui.properties.nodedefinition.NodeDefinitionVariablesContentProvider;
import com.tandbergtv.watchpoint.studio.ui.properties.nodedefinition.NodeDefinitionVariablesLabelProvider;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.MapNodeVariablesUtil;

/**
 * Composite for the start variables in a node definition. Defines a table in which variables can be
 * added and removed.
 * 
 * @author Imran Naqvi
 * 
 */
public class NodeDefinitionVariablesSectionComposite {

    protected TabbedPropertySheetWidgetFactory widgetFactory;
    protected Composite parent;

    protected Label messageLabel;
    private Button autoMapButton;
    private Table table;

    private String[] columnTitles = new String[] { "Name", "Template Variable", "Attachment", "Type", "Composite Key" };
    protected Composite composite;
    protected TableViewer tableViewer;
    private TableColumn attachmentColumn;
    private TableColumn keyColumn;
    private NodeDefinitionVariablesCellModifier cellModifier;
    private ExpressionEditor expressionEditor;
    private IStructuredContentProvider contentProvider;

    public NodeDefinitionVariablesSectionComposite(Composite parent, TabbedPropertySheetWidgetFactory widgetFactory) {
        this(parent, widgetFactory, null);
    }

    public NodeDefinitionVariablesSectionComposite(Composite parent, TabbedPropertySheetWidgetFactory widgetFactory, IStructuredContentProvider contentProvider) {
        this.parent = parent;
        this.widgetFactory = widgetFactory;
        this.contentProvider = contentProvider;

        createComponents();
    }

    protected void createComponents() {
        createComposite();
        createAutomapButton();
        createTable();
        createTableViewer();
    }

    protected void createComposite() {
        composite = widgetFactory.createFlatFormComposite(parent);
    }

    protected void createTable() {
        Composite exclusiveTableParent = widgetFactory.createFlatFormComposite(composite);

        int tableStyle = SWT.V_SCROLL | SWT.H_SCROLL | SWT.FULL_SELECTION;
        table = widgetFactory.createTable(exclusiveTableParent, tableStyle);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        FormData formData = new FormData();
        formData.top = new FormAttachment(messageLabel, 5);
        formData.left = new FormAttachment(0, 5);
        formData.bottom = new FormAttachment(100, 0);
        formData.right = new FormAttachment(100, -5);
        formData.width = 600;
        exclusiveTableParent.setLayoutData(formData);

        TableColumn nameColumn = new TableColumn(table, SWT.LEFT);
        nameColumn.setText(columnTitles[0]);
        TableColumn mappedNameColumn = new TableColumn(table, SWT.LEFT);
        mappedNameColumn.setText(columnTitles[1]);
        attachmentColumn = new TableColumn(table, SWT.CENTER);
        attachmentColumn.setText(columnTitles[2]);
        TableColumn typeColumn = new TableColumn(table, SWT.CENTER);
        typeColumn.setText(columnTitles[3]);
        keyColumn = new TableColumn(table, SWT.CENTER);
        keyColumn.setText(columnTitles[4]);

        TableColumnLayout layout = new TableColumnLayout();
        exclusiveTableParent.setLayout(layout);
        layout.setColumnData(nameColumn, new ColumnWeightData(20, 100));
        layout.setColumnData(mappedNameColumn, new ColumnWeightData(20, 100));
        layout.setColumnData(attachmentColumn, new ColumnWeightData(10, 60, false));
        layout.setColumnData(typeColumn, new ColumnWeightData(5, 60, false));
        layout.setColumnData(keyColumn, new ColumnWeightData(10, 60, false));
    }

    protected void createTableViewer() {
        tableViewer = new TableViewer(table);

        tableViewer.setUseHashlookup(true);
        tableViewer.setColumnProperties(columnTitles);
        if (contentProvider == null) {
            contentProvider = new NodeDefinitionVariablesContentProvider();
        }
        tableViewer.setContentProvider(contentProvider);
        tableViewer.setLabelProvider(new NodeDefinitionVariablesLabelProvider(attachmentColumn, keyColumn));

        CellEditor[] editors = new CellEditor[columnTitles.length];
        editors[0] = new TextCellEditor(table, SWT.BORDER);
        editors[1] = new CustomComboBoxCellEditor(table);
        editors[2] = new CheckboxCellEditor(table);
        editors[3] = new ComboBoxCellEditor(table, new String[0], SWT.BORDER);
        editors[4] = new CheckboxCellEditor(table);

        // instantiate expressionEditor
        expressionEditor = new ExpressionEditor(tableViewer, 1);

        // Assign the cell editors to the viewer 
        tableViewer.setCellEditors(editors);

        cellModifier = new NodeDefinitionVariablesCellModifier(tableViewer, columnTitles);

        // Set the cell modifier for the viewer
        tableViewer.setCellModifier(cellModifier);

        TableViewerColumn viewerColumn = new TableViewerColumn(tableViewer, attachmentColumn);
        viewerColumn.setLabelProvider((CellLabelProvider) tableViewer.getLabelProvider());
        viewerColumn = new TableViewerColumn(tableViewer, keyColumn);
        viewerColumn.setLabelProvider((CellLabelProvider) tableViewer.getLabelProvider());

        table.pack();
    }

    protected void createAutomapButton() {
        messageLabel = widgetFactory.createLabel(composite, "Define the used variables :");
        FormData formData = new FormData();
        formData.top = new FormAttachment(0, 5);
        formData.left = new FormAttachment(0, 0);
        messageLabel.setLayoutData(formData);

        autoMapButton = widgetFactory.createButton(composite, "Auto Map Variables", SWT.PUSH);
        formData = new FormData();
        formData.top = new FormAttachment(0, 0);
        formData.left = new FormAttachment(messageLabel, 5);
        formData.width = 150;
        autoMapButton.setLayoutData(formData);

        autoMapButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                NodeDefinition nodeDefinition = (NodeDefinition) tableViewer.getInput();
                MapNodeVariablesUtil.mapNodeVariables(Utility.getTemplate(nodeDefinition), nodeDefinition);

                expressionEditor.setContextVariables(Utility.getVariables((WPAbstractNode)nodeDefinition));
                expressionEditor.reinitialize();
                tableViewer.refresh();
            }
        });
    }

    public void setInput(NodeDefinition nodeDefinition, IProject project) {
        tableViewer.setInput(nodeDefinition);
        List<WPVariable> startStateVariables = Utility.getVariables((WPAbstractNode)nodeDefinition);
        ((ComboBoxViewerCellEditor) tableViewer.getCellEditors()[1]).setInput(Utility.toNameArray(startStateVariables));

        if (NodeDefinitionType.SuperState.equals(nodeDefinition.getNodeType())) {
            nodeDefinition.updateSuperStateRequiredVariables();
        }

        expressionEditor.setupAndInitialize(startStateVariables, project, new CustomExpression());
    }

    public class CustomExpression implements IExpression {
        private IValidatableElement model;

        @Override
        public void setModel(IValidatableElement model) {
            this.model = model;
        }

        @Override
        public IValidatableElement getModel() {
            return model;
        }

        @Override
        public String getText() {
            WPVariable variable = (WPVariable) model;
            return variable.getName() != null ? variable.getName() : "";
        }

        @Override
        public DataType getReturnType() {
            WPVariable variable = (WPVariable) model;
            
            if (variable.getType() == null)
            	return DataType.STRING;
            
            return DataType.valueOf(variable.getType());
        }

        @Override
        public boolean isValidatable() {
            WPVariable variable = (WPVariable) model;
            return variable.isReadable();
        }
    }

    public void setContentProvider(IStructuredContentProvider contentProvider) {
        this.contentProvider = contentProvider;
    }
}