/*
 * Created on Oct 29, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.properties;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.common.model.NamedElement;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.external.wpexport.impl.template.UniqueNodeNameGenerator;
import com.tandbergtv.watchpoint.studio.ui.editor.NodeDefinitionEditor;
import com.tandbergtv.watchpoint.studio.ui.editor.WatchPointTemplateEditor;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;

/**
 * Composite wrapper that displays the properties of a Named Element.
 * 
 * @author Vijay Silva
 */
public class WPNamedElementComposite implements FocusListener
{
	private TabbedPropertySheetWidgetFactory widgetFactory;

	private Composite wrappedComposite;

	private Label nameLabel;

	private Text nameText;

	private NamedElement namedElement;

	private AbstractPropertySection parentPropertySection;

	/**
	 * Constructor
	 * 
	 * @param parentSection
	 *            The Parent
	 * @param composite
	 *            The Composite wrapped by this Composite container
	 * @param toolkit
	 *            The Toolkit used to create the widgets on the Composite
	 */
	public WPNamedElementComposite(AbstractPropertySection parentSection, Composite composite,
			TabbedPropertySheetWidgetFactory toolkit)
	{
		this.parentPropertySection = parentSection;
		this.wrappedComposite = composite;
		this.widgetFactory = toolkit;

		this.initialize();
	}

	/*
	 * Initialize the Composite, creating all controls to display in the Composite
	 */
	private void initialize()
	{
		/* Create the Controls */
		this.nameLabel = this.widgetFactory.createLabel(this.wrappedComposite, "Name");
		this.nameText = widgetFactory.createText(this.wrappedComposite, "");

		/* Initialize the Layout */
		FormData data = null;

		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(0, 2);
		this.nameLabel.setLayoutData(data);

		data = new FormData();
		data.left = new FormAttachment(0, 105);
		data.top = new FormAttachment(0, 0);
		data.width = 300;
		this.nameText.setLayoutData(data);
	}

	/**
	 * Set the Named Element to display on the Composite
	 * 
	 * @param namedElement
	 *            The Named Element that needs to be displayed.
	 */
	public void setNamedElement(NamedElement namedElement)
	{
		this.unhookListeners();
		clearControls();

		this.namedElement = namedElement;
		if (namedElement != null)
		{
			updateControls();
			hookListeners();
		}
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/*
	 * Hook all the listeners associated with this composite
	 */
	private void hookListeners()
	{
		this.nameText.addFocusListener(this);
	}

	/*
	 * Unhook all the listeners associated with this composite
	 */
	private void unhookListeners()
	{
		this.nameText.removeFocusListener(this);
	}

	/*
	 * Clear the data displayed in this composite
	 */
	private void clearControls()
	{
		this.nameText.setText("");
		this.nameText.setEnabled(true);
		this.nameText.setTextLimit(Text.LIMIT);
	}

	/*
	 * Updates the data displayed on the controls in this composite
	 */
	private void updateControls()
	{
		String name = this.namedElement.getName();
		this.nameText.setText(name == null ? "" : name);

		boolean enabled = true;
		int textLimit = Text.LIMIT;
		IWorkbenchPart part = this.parentPropertySection.getPart();

		if (this.namedElement instanceof WorkflowTemplate)
		{
			enabled = false;
			textLimit = 50;
		}

		/* Special case handling for Node Definition Editor */
		else if (part instanceof NodeDefinitionEditor)
		{
			if (this.isReadOnlyNamedElement((NodeDefinitionEditor) part))
			{
				enabled = false;
				textLimit = 50;
			}
		}

		nameText.setEnabled(enabled);
		nameText.setTextLimit(textLimit);
	}

	/*
	 * Get the Name specified in the Name Text
	 */
	private String getNameText()
	{
		String text = nameText.getText();
		if ("".equals(text) && !namedElement.isNameMandatory())
		{
			text = null;
		}

		return text;
	}

	/*
	 * Checks if the Named Element cannot be renamed (only called when the part firing the selected
	 * element is a Node Definition Editor).
	 */
	private boolean isReadOnlyNamedElement(NodeDefinitionEditor editor)
	{
		NodeDefinition rootElement = editor.getNodeDefinition();
		if (rootElement == null)
			return false;

		NodeDefinitionType nodeType = rootElement.getNodeType();
		boolean isRootElement = (rootElement == this.namedElement);
		boolean isSingleNode = ((nodeType == NodeDefinitionType.MessageNode) && (rootElement.getNode() == this.namedElement));

		return (isRootElement || isSingleNode);
	}

	// ========================================================================
	// ===================== FOCUS EVENT HANDLING
	// ========================================================================

	/**
	 * @see org.eclipse.swt.events.FocusListener#focusGained(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusGained(FocusEvent e)
	{
	}

	/**
	 * @see org.eclipse.swt.events.FocusListener#focusLost(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusLost(FocusEvent e)
	{
		if (e.widget == nameText)
		{
			if (this.namedElement == null)
				return;

			/* Trim the Text */
			String value = this.nameText.getText().trim();
			
			if (this.namedElement.isNameMandatory() && "".equals(value))
			{
				Shell shell = Display.getCurrent().getActiveShell();
				MessageDialog.openError(shell, "Error", "The name cannot be blank.");
				this.nameText.setText(this.namedElement.getName());
				return;
			}
			if(this.namedElement instanceof NodeElement){
	
				/*Checking to see if there is another node with this name*/
				IWorkbenchPart part = this.parentPropertySection.getPart();
				NodeElementContainer container = null;
				if(part instanceof WatchPointTemplateEditor)
					container = ((WatchPointTemplateEditor)part).getProcessDefinition();
				else if(part instanceof NodeDefinitionEditor)
					container = ((NodeDefinitionEditor)part).getNodeDefinition();
				
				UniqueNodeNameGenerator nameHelper = new UniqueNodeNameGenerator(false);
				nameHelper.setRootContainer(container);
				
				if (nameHelper.nameClash(value, (NodeElement) namedElement)){
					Shell shell = Display.getCurrent().getActiveShell();
					MessageDialog.openError(shell, "Error", "There is already a node named " + value + ".");
					this.nameText.setText(this.namedElement.getName());
					return;
				}
			}
			if (!value.equals(this.nameText.getText()))
				this.nameText.setText(value);

			/* Update the Named Element's Name */
			namedElement.setName(getNameText());
		}
	}
}
