package com.tandbergtv.watchpoint.studio.ui.properties.template;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableEditor;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.jbpm.gd.common.model.GenericElement;
import org.jbpm.gd.jpdl.model.StartState;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.properties.WPExceptionHandlerContainerComposite;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.util.ValidationMessages;
import com.tandbergtv.watchpoint.studio.validation.IValidationService;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageType;
import com.tandbergtv.watchpoint.studio.validation.ValidationServiceFactory;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

public class ExceptionArgumentTableItemWrapper implements DisposeListener, FocusListener {
	
	private TableItem tableItem;

	private Text nameText;

	private Text valueText;

	private WPExceptionHandlerContainerComposite parent;

	private GenericElement genericElement;

	private IValidationService validationService;

	/**
	 * Constructor
	 * 
	 * @param parent
	 * @param table
	 * @param genericElement
	 * @param isEditable
	 */
	public ExceptionArgumentTableItemWrapper(WPExceptionHandlerContainerComposite parent, Table table,
			GenericElement genericElement, boolean isEditable) {
		this.parent = parent;
		this.genericElement = genericElement;
		this.validationService = ValidationServiceFactory.createFactory().createValidationService();

		initializeTableItem(table);
		initializeNameText(table);
		initializeValueText(table);
		nameText.setEnabled(isEditable);
	}

	public IValidationService getValidationService() {
		return validationService;
	}
	
	/**
	 * @see org.eclipse.swt.events.FocusListener#focusLost(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusLost(FocusEvent e) {
		if (e.widget == this.nameText) {
			if (genericElement.getName().equals(this.nameText.getText()))
				return;
			genericElement.setName(this.nameText.getText());
		}
		else if (e.widget == this.valueText) {
			//if (genericElement.getValue().equals(this.valueText.getText()))
				//return;
			if (isValueValid())
				genericElement.setValue(this.valueText.getText());
			else
				this.valueText.setText(this.genericElement.getValue());
		}
	}

	/**
	 * @see org.eclipse.swt.events.FocusListener#focusGained(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusGained(FocusEvent e) {
		if (e.widget == this.nameText || e.widget == this.valueText) {
			this.tableItem.getParent().setSelection(this.tableItem);
		}
	}

	/**
	 * @see org.eclipse.swt.events.DisposeListener#widgetDisposed(org.eclipse.swt.events.DisposeEvent)
	 */
	public void widgetDisposed(DisposeEvent e) {
		if (nameText == null || nameText.isDisposed())
			return;
		if (valueText == null || valueText.isDisposed())
			return;
		nameText.removeFocusListener(this);
		nameText.dispose();
		valueText.removeFocusListener(this);
		valueText.dispose();
	}

	/**
	 * Gets the table item.
	 * 
	 * @return table item for this item wrapper
	 */
	public TableItem getTableItem() {
		return tableItem;
	}

	/**
	 * Initializes the table item.
	 * 
	 * @param table
	 */
	protected void initializeTableItem(Table table) {
		tableItem = new TableItem(table, SWT.NONE);
		tableItem.addDisposeListener(this);
		tableItem.setData(this);
	}

	/**
	 * Initializes the mapped name text box.
	 * 
	 * @param table
	 *            the table in which the text box is being added
	 */
	protected void initializeNameText(Table table) {
		nameText = createText(table, 0);
		FormData data = new FormData();
		nameText.setLayoutData(data);
		nameText.addFocusListener(this);
		nameText.setText(getArgumentName());
		nameText.setEditable(true);
	}

	/**
	 * Initializes the Argument Value text box.
	 * 
	 * @param table
	 *            the table in which the text box is being added
	 */
	protected void initializeValueText(Table table) {
		valueText = createText(table, 1);
		FormData data = new FormData();
		valueText.setLayoutData(data);
		valueText.addFocusListener(this);
		valueText.setText(getArgumentValue());
		valueText.setEditable(true);
	}

	/**
	 * Creates a combo box in the given table in the column specified by the given index.
	 * 
	 * @param table
	 *            the table in which the combo box is being created
	 * @param index
	 *            index of the column in which to create the combo box
	 * @return the combo box created
	 */
	protected Text createText(Table table, int index) {
		TableEditor textEditor = new TableEditor(table);
		Text result = new Text(table, SWT.NORMAL);
		result.setVisible(true);
		result.pack();
		textEditor.minimumWidth = result.getSize().x;
		textEditor.horizontalAlignment = SWT.LEFT;
		textEditor.grabHorizontal = true;
		textEditor.setEditor(result, tableItem, index);
		return result;
	}

	/**
	 * Get the Argument name.
	 * 
	 * @return Argument name
	 */
	public String getArgumentName() {
		return genericElement.getName();
	}

	/**
	 * Get the Argument value.
	 * 
	 * @return Argument value
	 */
	public String getArgumentValue() {
		return genericElement.getValue();
	}

	/*
	 * Checks to see if the variable value is valid.
	 */
	private boolean isValueValid() {
		boolean valid = true;
		GenericElement element = new GenericElement();
		element.setName(this.nameText.getText());
		element.setValue(this.valueText.getText());
		String name = this.nameText.getText();
		ValidationMessage message = null;
		
	    if (name.equals("retry")) {
	        message = ValidationMessageAdder.getInstance().createValidationMessage(element, ValidationMessageCode.EXCEPTION_HANDLER_ACTION_ARGUMENT_VALUE1);
	    } else if (name.equals("email")) {
            message = ValidationMessageAdder.getInstance().createValidationMessage(element, ValidationMessageCode.EXCEPTION_HANDLER_ACTION_ARGUMENT_VALUE2);
	    } else if (name.matches("error.+Variable")) {
            message = ValidationMessageAdder.getInstance().createValidationMessage(element, ValidationMessageCode.EXCEPTION_HANDLER_ACTION_ARGUMENT_VALUE3);
	    }
		valid = this.isValid(message, name, this.valueText.getText());
		if (!valid) {
			this.displayValidationErrors(message);
		}

		return valid;
	}

	/* Check if the list of validation messages signals a validation error */
	private boolean isValid(ValidationMessage message, String name, String value) {
		if (message == null)
			return false;
		if (name.equals("email")) {
			if (value == null || value.length() == 0)
				return false;
			return true;
		}
		if (name.equals("retry")) {
			try {
				int i = Integer.parseInt(value);

				if (i > 0 && i <= 10)
					return true;
			} catch (NumberFormatException e) {
				return false;
			}
		}
		if (name.matches("error.+Variable")) {
			value = value.trim();
			if (value.isEmpty()) {
				return true;
			}
		    
			WorkflowTemplate template = (WorkflowTemplate) Utility.getNodeElementContainer(
					this.parent.getParentPropertySection().getPart());
		    
			StartState startState = template.getStartState();
			if (startState == null || startState.getTask() == null)
				return false;

			for (Variable element : startState.getTask().getController().getVariables()) {
				if (element.getName().equals(value)) {
					return true;
				}
			}
		}

		return false;
	}

	/*
	 * Display the list of validation messages provided in a single message dialog.
	 */
	private void displayValidationErrors(ValidationMessage message) {
		Shell shell = Display.getCurrent().getActiveShell();

		ValidationMessages validationMessages = null;

		try {
			validationMessages = ValidationMessages.getInstance();
		}
		catch (Exception e) {
			throw new UIException("Could not load the validation messages.", e);
		}

		MessageDialog.openError(shell, "Error", validationMessages.getMessage(message));
	}

	/**
	 * @return the parent
	 */
	public WPExceptionHandlerContainerComposite getParent() {
		return parent;
	}

	/*
	 * Checks to see if the given name already exists in the action's generic elements.
	 */
	/*private boolean checkDuplicateName(String name) {
		for (GenericElement element : this.parent.getTaskNode().getAction().getGenericElements()) {
			if (element.getName().equals(name))
				return true;
		}
		return false;
	}*/
}
