/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.view;

import java.util.List;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.part.ViewPart;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * Abstract tree view explorer which shows a list of IWatchPointDTO objects. The users can create,
 * view, edit and delete instances of these objects. The actions are added by this class but their
 * run methods must be defined by child classes.
 * 
 * @author Imran Naqvi
 * 
 */
public abstract class AbstractTreeViewExplorer extends ViewPart implements IDoubleClickListener,
		ISelectionChangedListener {

	private TreeViewer treeViewer;

	protected Action createAction, viewAction, editAction, deleteAction, refreshAction;

	/**
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) {
		/* Setting layout and layout data for parent */
		parent.setLayout(new GridLayout());
		parent.setLayoutData(getLayoutData());
		/* Creating tree panel */
		Composite treePanel = new Composite(parent, SWT.BORDER);
		treePanel.setLayout(new GridLayout(1, true));
		treePanel.setLayoutData(getTreePanelData());
		treePanel.pack();
		this.treeViewer = createTreeViewer(treePanel);
		parent.pack();
		/* Create actions and toolbar */
		createActions();
		createContextMenu();
		createToolbar();
		hookGlobalActions();
	}

	/**
	 * Get the TreeViewer component
	 * 
	 * @return The TreeViewer Component
	 */
	protected TreeViewer getTreeViewer() {
		return this.treeViewer;
	}

	/**
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	@Override
	public void setFocus() {
		this.treeViewer.getControl().setFocus();
	}

	/**
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event) {
	}

	/**
	 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
	 */
	public void doubleClick(DoubleClickEvent event) {
		edit();
	}

	/**
	 * Refreshes the list of objects being shown in this tree viewer.
	 */
	public void refresh() {
		this.treeViewer.getTree().setRedraw(false);
		try {
			Object[] expandedItems = this.getTreeViewer().getExpandedElements();
			treeViewer.setInput(getInput());
			for (int i = 0; i < expandedItems.length; i++) {
				Object item = expandedItems[i];
				if(item instanceof ResourceType){
					this.treeViewer.expandToLevel(item, 1);
				}
				if(item instanceof NodeDefinition){
					this.treeViewer.expandToLevel(item, 1);
				}
			}
		} finally {
			this.treeViewer.getTree().setRedraw(true);
		}
	}
	
	public void expandElement(Object element){
		this.treeViewer.getTree().setRedraw(false);
		try {
			this.treeViewer.expandToLevel(element, 1);
		} finally {
			this.treeViewer.getTree().setRedraw(true);
		}
	}

	protected void createToolbar() {
		IToolBarManager tbManager = getViewSite().getActionBars().getToolBarManager();
		tbManager.add(createAction);
		tbManager.add(viewAction);
		tbManager.add(editAction);
		tbManager.add(deleteAction);
		tbManager.add(refreshAction);
	}

	/*
	 * Fills context menus for the nodes in the tree viewer.
	 * 
	 * @param mgr menu manager of the treeviewer
	 */
	protected void fillContextMenu(IMenuManager mgr) {
		mgr.add(viewAction);
		mgr.add(editAction);
		mgr.add(deleteAction);
	}

	/**
	 * Initializes the actions and their run methods.
	 */
	protected void createActions() {
		/* Create 'Create' Action */
		createAction = new ViewAction(this, "Create") {
			public void performAction() {
				create();
			}
		};
		createAction.setImageDescriptor(Utility.getImageDescriptor("create.png"));

		/* Create 'View' Action */
		viewAction = new ViewAction(this, "View") {
			public void performAction() {
				view();
			}
		};
		viewAction.setImageDescriptor(Utility.getImageDescriptor("view.png"));

		/* Create 'Edit' Action */
		editAction = new ViewAction(this, "Edit") {
			public void performAction() {
				edit();
			}
		};
		editAction.setImageDescriptor(Utility.getImageDescriptor("edit.png"));

		/* Create 'Delete' Action */
		deleteAction = new ViewAction(this, "Delete") {
			public void performAction() {
				delete();
			}
		};
		deleteAction.setImageDescriptor(Utility.getImageDescriptor("delete.png"));

		/* Create 'Refresh' Action */
		refreshAction = new ViewAction(this, "Refresh") {
			public void performAction() {
				refresh();
			}
		};
		refreshAction.setImageDescriptor(Utility.getImageDescriptor("refresh.png"));
	}

	/**
	 * @return the currently selected object in the tree viewer
	 */
	protected ITreeSelection getSelection() {
		return (ITreeSelection) this.treeViewer.getSelection();
	}

	/**
	 * Gets a list of IWatchPointDTO objects to show in the tree viewer.
	 * 
	 * @return list of IWatchPointDTO objects
	 */
	protected abstract List<? extends IWatchPointDTO> getInput();

	/**
	 * Creates an instance of the IWatchPointDTO object being shown by this tree viewer.
	 */
	protected abstract void create();

	/**
	 * Opens the IWatchPointDTO object selected in this tree viewer for view.
	 */
	protected abstract void view();

	/**
	 * Opens the IWatchPointDTO object selected in this tree viewer for edit.
	 */
	protected abstract void edit();

	/**
	 * Deletes the IWatchPointDTO object selected in this tree viewer.
	 */
	protected abstract void delete();

	/**
	 * Gets the content provider for this tree viewer.
	 */
	protected abstract IContentProvider getContentProvider();

	/**
	 * Gets the label provider for this tree viewer.
	 */
	protected abstract IBaseLabelProvider getLabelProvider();

	/*
	 * Gets the layout data for the tree view.
	 * 
	 * @return
	 */
	private GridData getLayoutData() {
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true, 5, 5);
		data.minimumHeight = 300;
		return data;
	}

	/*
	 * Gets the layout data for the tree panel. @return
	 */
	private GridData getTreePanelData() {
		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true);
		return data;
	}

	/*
	 * Hooks global actions to toolbar actions.
	 */
	protected void hookGlobalActions() {
		IActionBars bars = getViewSite().getActionBars();
		bars.setGlobalActionHandler(ActionFactory.DELETE.getId(), deleteAction);
	}

	/*
	 * Creates the tree viewer. @param treePanel the panel in which to create the treeviewer
	 */
	protected TreeViewer createTreeViewer(Composite treePanel) {
		TreeViewer viewer = new TreeViewer(treePanel, SWT.SINGLE);
		viewer.setContentProvider(getContentProvider());
		viewer.setLabelProvider(getLabelProvider());
		viewer.setInput(getInput());
		viewer.getTree().setLayout(new GridLayout());
		viewer.getTree().setLayoutData(
				new GridData(GridData.FILL_BOTH | SWT.H_SCROLL | SWT.V_SCROLL));
		viewer.addDoubleClickListener(this);
		viewer.addSelectionChangedListener(this);
		viewer.setComparator(new ViewerComparator());
		return viewer;
	}

	/*
	 * Creates context menus for the nodes in the tree viewer based on the currently selected node.
	 */
	protected void createContextMenu() {
		// Create menu manager.
		MenuManager menuMgr = new MenuManager();
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager mgr) {
				fillContextMenu(mgr);
			}
		});

		// Create menu.
		Menu menu = menuMgr.createContextMenu(treeViewer.getControl());
		treeViewer.getControl().setMenu(menu);
	}
}
