/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.view;

import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;
import org.jbpm.gd.common.model.NamedElement;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.util.ValidationMessages;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageType;

/**
 * Label provider class for the validation errors view.
 * 
 * @author Imran Naqvi
 * 
 */
public class ValidationLabelProvider extends LabelProvider implements ITableLabelProvider {

	private static final String ERROR_IMAGE = "validation_error.png";

	private static final String WARNING_IMAGE = "validation_warning.png";

	private static final String NOTIFICATION_IMAGE = "validation_notification.png";

	private ImageRegistry imageRegistry;

	/**
	 * Create the label provider.
	 */
	public ValidationLabelProvider() {
		this.imageRegistry = new ImageRegistry();

		String key = ValidationMessageType.Error.name();
		this.imageRegistry.put(key, Utility.getImageDescriptor(ERROR_IMAGE));

		key = ValidationMessageType.Warning.name();
		this.imageRegistry.put(key, Utility.getImageDescriptor(WARNING_IMAGE));

		key = ValidationMessageType.Notification.name();
		this.imageRegistry.put(key, Utility.getImageDescriptor(NOTIFICATION_IMAGE));
	}

	/**
	 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
	 */
	public Image getColumnImage(Object element, int columnIndex) {
		if (columnIndex != 0 || !(element instanceof ValidationMessage))
			return null;

		ValidationMessage message = (ValidationMessage) element;
		return this.getMessageImage(message.getType());
	}

	/*
	 * Get the Image for the given validation message type
	 */
	private Image getMessageImage(ValidationMessageType messageType) {
		if (messageType == null)
			return null;

		return this.imageRegistry.get(messageType.name());
	}

	/**
	 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
	 */
	public String getColumnText(Object element, int columnIndex) {
		String columnText = "";

		if (element instanceof ValidationMessage) {
			ValidationMessage message = (ValidationMessage) element;
			switch (columnIndex) {
				case 0:
					try {
						columnText = ValidationMessages.getInstance().getMessage(message);
					}
					catch (Exception e) {
						throw new UIException("Could not read validation message "
								+ message.getCode() + " from property file.", e);
					}
					break;

				case 1:
					if (message.getElement() instanceof NamedElement)
						columnText = ((NamedElement) message.getElement()).getName();
					else if (message.getElement() instanceof IWatchPointDTO)
						columnText = new DTONameDecorator((IWatchPointDTO)message.getElement()).getName();
					break;

				case 2:
					if (message.getElement() instanceof NamedElement)
						columnText = ((NamedElement) message.getElement()).getLabel();
					else if (message.getElement() instanceof IWatchPointDTO)
						columnText = new DTONameDecorator((IWatchPointDTO)message.getElement()).getType();
					break;
			}
		}

		else if (element instanceof ValidationMessageGroup && columnIndex == 0) {
			ValidationMessageGroup group = (ValidationMessageGroup) element;
			int count = (group.getMessages() != null) ? group.getMessages().size() : 0;
			columnText = this.getDisplayText(group.getMessageType()) + " (" + count + " items)";
		}

		return columnText;
	}

	/*
	 * Get the Group Name to display for the Message Type
	 */
	private String getDisplayText(ValidationMessageType messageType) {
		String text = "Undefined";

		if (messageType == null)
			return text;

		switch (messageType) {
			case Error:
				text = "Errors";
				break;

			case Warning:
				text = "Warnings";
				break;

			case Notification:
				text = "Notifications";
				break;
		}

		return text;
	}

	/**
	 * @see org.eclipse.jface.viewers.BaseLabelProvider#isLabelProperty(java.lang.Object,
	 *      java.lang.String)
	 */
	@Override
	public boolean isLabelProperty(Object element, String property) {
		return false;
	}

	/**
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#dispose()
	 */
	public void dispose() {
		super.dispose();

		if (this.imageRegistry != null) {
			this.imageRegistry.dispose();
			this.imageRegistry = null;
		}
	}
}
