/**
 * ResourceTypeDataProvider.java
 * Created Feb 16, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.wizard.resource;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.jbpm.gd.jpdl.Logger;
import org.jbpm.gd.jpdl.Plugin;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.external.fs.dao.DataAccessFactoryImpl;
import com.tandbergtv.watchpoint.studio.external.imports.ImportDataProvider;
import com.tandbergtv.watchpoint.studio.external.jpf.JPFContextFactory;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.service.impl.ServiceImpl;

/**
 * @author Sahil Verma
 */
public class ResourceTypeDataProvider implements ImportDataProvider {
	
	private String location;
	
	private Collection<ResourceType> existing;
	
	private static final org.apache.log4j.Logger logger =
		org.apache.log4j.Logger.getLogger(ResourceTypeDataProvider.class);

	public ResourceTypeDataProvider() {
		super();
	}

	/**
	 * {@inheritDoc}
	 */
	public String getImportLocation() {
		return this.location;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setImportLocation(String location) {
		this.location = location;
	}

	/**
	 * {@inheritDoc}
	 */
	public Collection<? extends IWatchPointDTO> getItems() {
		if (this.location == null)
			throw new IllegalArgumentException("Import location must be specified");
		
		logger.debug("Importing from " + this.location);
		
		this.existing = ServiceFactory.createFactory().createResourceTypeService().getResourceTypeList();
		
		List<ResourceType> types = getTypesToImport();
		
		filter(types);
		
		logger.debug("Found " + types.size() + " items");
		
		return types;
	}

	/**
	 * @param types
	 */
	protected void filter(List<ResourceType> types) {
		Iterator<ResourceType> i = types.iterator();
		String pluginId = Plugin.getDefault().getBundle().getSymbolicName();
		MultiStatus status = new MultiStatus(pluginId, IStatus.WARNING, "Some items cannot be imported", null);
		
		while (i.hasNext()) {
			ResourceType rt = i.next();
			ResourceType existing = findExistingResourceType(rt.getSystemId());
			
			logger.debug(rt.getName() + " - v" + rt.getVersion() + ", looking for existing version");
			
			if (existing != null)
				logger.debug(existing.getName() + " - v" + existing.getVersion() + " exists");
			
			/* Ignore the resource type if we already have a more recent version */
			if (existing != null && existing.getVersion() >= rt.getVersion()) {
				i.remove();
				String message = rt.getName() + " v" + rt.getVersion() + " - version must exceed " + 
					existing.getVersion();
				status.add(new Status(IStatus.ERROR, pluginId, message));
				
				continue;
			}
		}
		
		if (status.getChildren().length > 0)
			Logger.log(status);
	}
	
	protected ResourceType findExistingResourceType(String systemId) {
		for (ResourceType rt : this.existing) {
			if (rt.getSystemId().equals(systemId))
				return rt;
		}
		
		return null;
	}
	
	protected List<ResourceType> getTypesToImport() {
		IResourceTypeService service = ServiceFactory.createFactory().createResourceTypeService();
		
		((ServiceImpl) service).setDataAccessFactory(new DataAccessFactoryImpl());
		((ServiceImpl) service).setPersistenceContextFactory(new JPFContextFactory(this.location));
		
		return service.getResourceTypeList();
	}
}
