/*******************************************************************************
 * Copyright (c) 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Bjorn Freeman-Benson - initial API and implementation
 *******************************************************************************/
package com.tandbergtv.watchpoint.studio.debugger.model;

import org.eclipse.core.resources.IFile;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IThread;
import org.eclipse.debug.core.model.IValue;
import org.eclipse.debug.ui.IDebugEditorPresentation;
import org.eclipse.debug.ui.IDebugModelPresentation;
import org.eclipse.debug.ui.IValueDetailListener;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.part.FileEditorInput;

import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * Renders Jpdl debug elements
 */
public class JpdlModelPresentation extends LabelProvider implements IDebugModelPresentation, IDebugEditorPresentation {
    /* (non-Javadoc)
     * @see org.eclipse.debug.ui.IDebugModelPresentation#setAttribute(java.lang.String, java.lang.Object)
     */
    public void setAttribute(String attribute, Object value) {
    }

    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
     */
    public Image getImage(Object element) {
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.ILabelProvider#getText(java.lang.Object)
     */
    public String getText(Object element) {
        try {
            StringBuilder text = new StringBuilder();
            if (element instanceof NodeBreakpoint) {
                NodeBreakpoint breakpoint = (NodeBreakpoint) element;
                text.append(breakpoint.getResourceName());
                text.append(" - ");
                text.append(breakpoint.getNodeName());
            } else if (element instanceof JpdlValue) {
                text.append(decorateValue((JpdlValue) element));
            } else if (element instanceof JpdlDebugTarget) {
                text.append(((JpdlDebugTarget) element).getName());
            } else if (element instanceof JpdlThread) {
                text.append(((JpdlThread) element).getName());
            } else if (element instanceof JpdlStackFrame) {
                text.append(((JpdlStackFrame) element).getNodeName());
            }
            return text.toString();
        } catch (DebugException e) {
            throw new RuntimeException(e);
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.debug.ui.IDebugModelPresentation#computeDetail(org.eclipse.debug.core.model.IValue, org.eclipse.debug.ui.IValueDetailListener)
     */
    public void computeDetail(IValue value, IValueDetailListener listener) {
        StringBuilder detail = null;
        try {
            detail = decorateValue(value);
        } catch (DebugException e) {
        }
        listener.detailComputed(value, detail.toString());
    }

    private StringBuilder decorateValue(IValue value) throws DebugException {
        StringBuilder detail = new StringBuilder();
        if (value.getReferenceTypeName().contains(String.class.getName())) {
            detail.append("\"").append(value.getValueString()).append("\"");
        } else if (value.getReferenceTypeName().contains(Integer.class.getName())) {
            detail.append(value.getValueString());
        } else {
            detail.append("[");
            detail.append(value.getValueString());
            detail.append("]");
        }
        return detail;
    }

    /* (non-Javadoc)
     * @see org.eclipse.debug.ui.ISourcePresentation#getEditorInput(java.lang.Object)
     */
    public IEditorInput getEditorInput(Object element) {
        if (element instanceof IFile) {
            return new FileEditorInput((IFile) element);
        }
        if (element instanceof NodeBreakpoint) {
            return new FileEditorInput((IFile) ((NodeBreakpoint) element).getMarker().getResource());
        }
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.debug.ui.ISourcePresentation#getEditorId(org.eclipse.ui.IEditorInput, java.lang.Object)
     */
    public String getEditorId(IEditorInput input, Object element) {
        return "org.jbpm.gd.jpdl.editor";
    }

    @Override
    public boolean addAnnotations(IEditorPart editorPart, IStackFrame frame) {
        boolean addedAnnotation = false;
        for (NodeDefinition nodeDefinition : Utility.getNodeDefinitions(editorPart)) {
            if (nodeDefinition.getName().equals(((JpdlStackFrame) frame).getNodeName())) {
                nodeDefinition.setHighlight(true);
                addedAnnotation = true;
            } else {
                nodeDefinition.setHighlight(false);
            }
            Utility.refreshEditPartByNodeName(editorPart, nodeDefinition.getName());
        }
        return addedAnnotation;
    }

    @Override
    public void removeAnnotations(IEditorPart editorPart, IThread thread) {
        for (NodeDefinition nodeDefinition : Utility.getNodeDefinitions(editorPart)) {
            nodeDefinition.setHighlight(false);
            Utility.refreshEditPartByNodeName(editorPart, nodeDefinition.getName());
        }
    }
}