/**
 * ActionInserter.java
 * Created Feb 29, 2012
 */
package com.tandbergtv.watchpoint.studio.external.wpexport.impl.template;

import static com.tandbergtv.watchpoint.studio.ui.model.TemplateActionRefNames.RESOURCE_RELEASE_ACTION;
import static com.tandbergtv.watchpoint.studio.ui.model.TemplateActionRefNames.RESOURCE_REQUEST_ACTION;
import static com.tandbergtv.watchpoint.studio.util.SemanticElementUtil.TRANSITION_PATH_SEPARATOR;

import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.jpdl.model.Action;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.ui.model.IResourceGroupElement;
import com.tandbergtv.watchpoint.studio.ui.model.ResourceGroup;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.model.WPTransition;

/**
 * Inserts actions at various graph elements, as appropriate
 */
public class ActionInserter {
	
	/**
	 * @param parent
	 */
	public static void addActions(NodeElementContainer parent) {
		/* Be sure to add resource-release action first */
		addResourceReleaseActions(parent);
		addResourceRequestActions(parent);
	}
	
	// ========================================================================
	// ===================== RESOURCE REQUEST AND RELEASE ACTION HANDLING
	// ========================================================================

	/*
	 * Adds resource release actions to transitions depending on whether the transition sources need
	 * a resource.
	 */
	private static void addResourceReleaseActions(NodeElementContainer parent) {
		for (NodeElement node : parent.getNodeElements()) {
			if (node instanceof IResourceGroupElement) {
				ResourceGroup resourceGroup = ((IResourceGroupElement) node).getResourceGroup();
				if (resourceGroup == null || resourceGroup.getName() == null
						|| resourceGroup.getName().length() == 0)
					continue;

				/* Add Resource Release action */
				if (node.getTransitions().length == 0)
					continue;
				
				Action action = createAction(node.getTransitions()[0].getFactory());
				action.setRefName(RESOURCE_RELEASE_ACTION);
				addRefAction((WPTransition) node.getTransitions()[0], action);
			} else if (node instanceof NodeElementContainer) {
				addResourceReleaseActions((NodeElementContainer) node);
			}
		}
	}

	/*
	 * Adds resource request actions to transitions depending on whether the transition targets need
	 * a resource.
	 */
	private static void addResourceRequestActions(NodeElementContainer parent) {
		for (NodeElement node : parent.getNodeElements()) {
			if (node instanceof IResourceGroupElement) {
				ResourceGroup resourceGroup = ((IResourceGroupElement) node).getResourceGroup();
				
				if (resourceGroup == null || resourceGroup.getName() == null || resourceGroup.getName().length() == 0)
					continue;
	
				/* Add Resource Request actions to incoming transitions */
				for (NodeElement currentNode : parent.getNodeElements()) {
					for (Transition transition : currentNode.getTransitions()) {
						if (transition.getTo().equals(node.getName()) || 
							transition.getTo().equals(TRANSITION_PATH_SEPARATOR + node.getName())) {
							Action action = createAction(transition.getFactory());
							action.setRefName(RESOURCE_REQUEST_ACTION);
							addRefAction((WPTransition) transition, action);
						}
					}
				}
			} else if (node instanceof NodeElementContainer) {
				addResourceRequestActions((NodeElementContainer) node);
			}
		}
	}

	/*
	 * Adds an action to the transition with the given ref name, if action doesn't have ref name.
	 */
	private static void addRefAction(WPTransition transition, Action action) {
		if (!transition.hasAction(action.getRefName()))
			transition.addActionElement(action);
	}

	/*
	 * Creates a new instance of Action
	 */
	private static Action createAction(SemanticElementFactory factory) {
		return (Action) factory.createById(SemanticElementConstants.ACTION_SEID);
	}
}
