/*
 * Created on Nov 14, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.actionconfig;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

/**
 * Singleton that managed the Action Class Configuration.
 * 
 * @author Vijay Silva
 */
public final class ActionConfigurationManager {

	private static final Logger logger = Logger.getLogger(ActionConfigurationManager.class);

	private static ActionConfigurationManager instance = new ActionConfigurationManager();

	private Map<String, ActionClassConfiguration> actionClasses;

	/**
	 * Singleton access method.
	 * 
	 * @return The singleton instance
	 */
	public static ActionConfigurationManager getInstance() {
		return instance;
	}

	/* Private Constructor */
	private ActionConfigurationManager() {
		this.initialize();
	}

	/**
	 * Get the Action Class Configuration with the matching class name
	 * 
	 * @param className
	 *            The Action Class name
	 * 
	 * @return the matching ActionClassConfiguration object
	 */
	public synchronized ActionClassConfiguration getActionClass(String className) {
		if (this.actionClasses != null && className != null)
			return actionClasses.get(className);

		return null;
	}

	/**
	 * Get all the Action Classes in a list sorted by class names.
	 * 
	 * @return The list of action classes sorted by class names.
	 */
	public synchronized List<ActionClassConfiguration> getAllActionClasses() {
		List<ActionClassConfiguration> classes = new ArrayList<ActionClassConfiguration>();
		if (this.actionClasses != null) {
			classes.addAll(actionClasses.values());
			Collections.sort(classes, new ActionClassComparator());
		}

		return classes;
	}

	/* Initialize the action class configuration */
	private void initialize() {
		this.actionClasses = null;

		try {
			ActionConfigurationReader reader = new ActionConfigurationReader();
			this.actionClasses = reader.read();
		}
		catch (ActionConfigurationException ace) {
			String msg = "Failed to read the Action Class Configuration, no pre-configured Action Classes available.";
			logger.error(msg, ace);
		}
	}

	/*
	 * Internal Comparator to sort the classes by class name.
	 */
	private class ActionClassComparator implements Comparator<ActionClassConfiguration> {

		/**
		 * Compares two ActionClassConfiguration object using the class name. Does not support null
		 * objects or null class names.
		 * 
		 * @param o1
		 *            The ActionClassConfiguration object to compare
		 * @param o2
		 *            The ActionClassConfiguration object to compare
		 * 
		 * @return a negative integer, zero, or a positive integer as the first argument is less
		 *         than, equal to, or greater than the second
		 * 
		 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
		 */
		public int compare(ActionClassConfiguration o1, ActionClassConfiguration o2) {
			return o1.getClassName().compareTo(o2.getClassName());
		}
	}
}
