/*
 * Created on Sep 24, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpage;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.widgets.Section;

import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.ResourceTypeEditor;
import com.tandbergtv.watchpoint.studio.ui.model.IEditableElement;

/**
 * Abstract implementation of a Form Page specifically for the ResourceType Form Pages.
 * 
 * @author Vijay Silva
 */
public abstract class ResourceTypeFormPage extends FormPage
{
	/**
	 * @param editor
	 *            The parent editor
	 * @param pageId
	 *            The page id
	 * @param pageTitle
	 *            The page title
	 */
	public ResourceTypeFormPage(ResourceTypeEditor editor, String pageId, String pageTitle)
	{
		super(editor, pageId, pageTitle);
	}

	// ========================================================================
	// ===================== INITIALIZING USER CONTROLS
	// ========================================================================

	/**
	 * @see org.eclipse.ui.forms.editor.FormPage#createFormContent(org.eclipse.ui.forms.IManagedForm)
	 */
	@Override
	protected void createFormContent(IManagedForm managedForm)
	{
		super.createFormContent(managedForm);

		// Create the Controls and initialize the Layouts
		this.createPageControls();
		this.initializePageLayout();

		// Set the form data
		this.populatePageData();
		this.setWidgetPermissions();

		managedForm.getForm().pack();
	}

	/**
	 * Create the Controls (including the Form Parts) to display on this page
	 */
	protected abstract void createPageControls();

	/**
	 * Specify the Layout for the Page and its Controls
	 */
	protected abstract void initializePageLayout();

	/**
	 * Populate the data to be displayed on the Page
	 */
	protected void populatePageData()
	{
		this.getManagedForm().setInput(this.getEditorInput());
	}
	
	/**
	 * Set the editor input, then the editor part can be updated with a new resource type
	 * @param input Editor Input
	 */
	public void setEditorInput(IEditorInput input) {
		super.setInput(input);
	}

	/**
	 * Set the permission for Widgets that are not part of any Form Part. Does nothing, by default.
	 */
	protected void setWidgetPermissions()
	{
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/**
	 * Get the Default Style to use when creating a Section
	 * 
	 * @return the Default Style bits used when creating a Section in a Resource Page
	 */
	public static int getDefaultSectionStyle()
	{
		return Section.CLIENT_INDENT | Section.TITLE_BAR | Section.EXPANDED | Section.COMPACT;
	}

	/**
	 * Creates a Grid Layout with a default margin of 10 pixels above and below.
	 * 
	 * @param columns
	 *            the number of columns
	 * @param equalColumnSize
	 *            flag to indicate if the column sizes are equal
	 * 
	 * @return The Grid Layout
	 */
	protected GridLayout createGridLayout(int columns, boolean equalColumnSize)
	{
		GridLayout gridLayout = new GridLayout(columns, equalColumnSize);
		gridLayout.marginLeft = gridLayout.marginRight = 0;
		gridLayout.marginTop = gridLayout.marginBottom = 0;
		gridLayout.marginHeight = gridLayout.marginWidth = 10;
		gridLayout.verticalSpacing = 15;

		return gridLayout;
	}

	/**
	 * Creates a GridData object that ensures the section horizontally fills the column, and grabs
	 * any horizontal space.
	 * 
	 * @return The GridData
	 */
	protected GridData createSectionGridData()
	{
		return new GridData(SWT.FILL, SWT.BEGINNING, true, false);
	}

	/**
	 * Get the ResourceType in the Editor Input
	 */
	protected ResourceType getResourceType()
	{
		IEditorInput input = this.getEditorInput();
		return (input != null) ? (ResourceType) input.getAdapter(ResourceType.class) : null;
	}

	/**
	 * Get the boolean flag to indicate if the editor input can be edited
	 */
	protected boolean isInputEditable()
	{
		boolean isEditable = true;

		IEditorInput input = this.getEditorInput();
		if (input != null)
		{
			IEditableElement editableElement = (IEditableElement) input.getAdapter(IEditableElement.class);
			if (editableElement != null)
				isEditable = editableElement.canEdit();
		}

		return isEditable;
	}
}
