/*
 * Created on Sep 24, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.SectionPart;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;

import com.tandbergtv.watchpoint.studio.dto.ConnectionType;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.ResourceTypeEditorInput;
import com.tandbergtv.watchpoint.studio.ui.model.IEditableElement;

/**
 * A SectionPart that is used to display ResourceType information.
 * 
 * @author Vijay Silva
 */
abstract class ResourceTypeSectionPart extends SectionPart
{
	private static final Logger logger = Logger.getLogger(ResourceTypeSectionPart.class);

	protected Composite sectionClient;

	protected IAdaptable input;

	protected boolean allowMarkDirty = false;

	protected IEditorPart editorPart;

	// ========================================================================
	// ===================== CONSTRUCTORS
	// ========================================================================

	/**
	 * @param parentPart
	 *            The Part that contains this composite
	 * @param parent
	 *            The Parent Composite
	 * @param toolkit
	 *            The toolkit to create the widgets
	 * @param style
	 *            The Style bits to create the Section
	 */
	public ResourceTypeSectionPart(IEditorPart parentPart, Composite parent, FormToolkit toolkit,
			int style)
	{
		super(parent, toolkit, style);
		this.editorPart = parentPart;
	}

	// ========================================================================
	// ===================== OVERRIDDEN METHODS
	// ========================================================================

	/**
	 * @see org.eclipse.ui.forms.AbstractFormPart#initialize(org.eclipse.ui.forms.IManagedForm)
	 */
	@Override
	public void initialize(IManagedForm form)
	{
		super.initialize(form);

		/* Build the Controls */
		this.createSectionControls(form);
		this.initializeSectionLayout(form);

		/* Accept any dirty events */
		this.allowMarkDirty = true;
	}

	/**
	 * @see org.eclipse.ui.forms.AbstractFormPart#refresh()
	 */
	@Override
	public void refresh()
	{
		super.refresh();

		// Need to refresh the contents of the widget, no dirty events need to be registered
		this.allowMarkDirty = false;

		// Need to update the state displayed in the UI
		this.populateSectionData();
		this.setWidgetPermissions();

		this.allowMarkDirty = true;
	}

	/**
	 * @see org.eclipse.ui.forms.AbstractFormPart#markDirty()
	 */
	@Override
	public void markDirty()
	{
		if (this.allowMarkDirty)
			super.markDirty();
	}

	/**
	 * @see org.eclipse.ui.forms.AbstractFormPart#setFormInput(java.lang.Object)
	 */
	@Override
	public boolean setFormInput(Object input)
	{
		if (input instanceof ResourceTypeEditorInput)
		{
			this.input = (ResourceTypeEditorInput) input;
			this.markStale();

			return true;
		}

		return false;
	}

	// ========================================================================
	// ===================== USER CONTROL INITIALIZATION
	// ========================================================================

	/**
	 * Create the Controls displayed in the composite
	 * 
	 * @param managedForm
	 *            The ManagedForm that will contain this Form Part
	 */
	protected void createSectionControls(IManagedForm managedForm)
	{
		FormToolkit toolkit = managedForm.getToolkit();
		Section section = this.getSection();
		this.sectionClient = toolkit.createComposite(section, SWT.WRAP);
		toolkit.paintBordersFor(this.sectionClient);
		section.setClient(this.sectionClient);

		this.createSectionClientControls(managedForm);
	}

	/**
	 * Create the widgets that need to be displayed in the section client.
	 * 
	 * @param managedForm
	 */
	protected abstract void createSectionClientControls(IManagedForm managedForm);

	/**
	 * Initialize the Layout for the Section Client and its child widgets
	 * 
	 * @param managedForm
	 *            The Managed Form that contains this form Part
	 */
	protected void initializeSectionLayout(IManagedForm managedForm)
	{
		/* Set the Layout Data for the Section Client */
		this.sectionClient.setLayoutData(this.createGridData());

		this.initializeSectionClientLayout(managedForm);
	}

	/**
	 * Initializes the Layout for the Section Client and all its Child Widgets
	 * 
	 * @param managedForm
	 *            The Managed Form containing this Part
	 */
	protected abstract void initializeSectionClientLayout(IManagedForm managedForm);

	/**
	 * Create a Basic Grid Layout (for the client).
	 * 
	 * @param columnCount
	 *            number of columns
	 * @param makeColumnsEqualWidth
	 *            flag to indicate if the columns have to be equal size
	 * 
	 * @return A Grid layout with 2 columns
	 */
	protected GridLayout createGridLayout(int columnCount, boolean makeColumnsEqualWidth)
	{
		GridLayout layout = new GridLayout(columnCount, makeColumnsEqualWidth);
		layout.marginLeft = layout.marginRight = 2;
		layout.marginTop = layout.marginBottom = 2;

		return layout;
	}

	/**
	 * Helper method to create a basic Grid Data element
	 */
	protected GridData createGridData()
	{
		return new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false);
	}

	/**
	 * Helper method to create a basic Grid Data element for labels
	 */
	protected GridData createLabelGridData()
	{
		GridData data = new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false);
		data.verticalIndent = 2;
		return data;
	}

	// ========================================================================
	// ===================== DISPLAYING THE INPUT DATA
	// ========================================================================

	/*
	 * Display the Input data in the Section Widgets
	 */
	protected abstract void populateSectionData();

	/*
	 * Set the permissions of the section widgets based on the input
	 */
	protected abstract void setWidgetPermissions();

	// ========================================================================
	// ===================== INPUT ACCESS METHODS
	// ========================================================================

	/*
	 * Get the ResourceType in the editor input
	 */
	protected ResourceType getResourceType()
	{
		return (this.input != null) ? (ResourceType) this.input.getAdapter(ResourceType.class)
				: null;
	}

	/*
	 * Get the Message in the editor input
	 */
	protected Message getMessage()
	{
		return (this.input != null) ? (Message) this.input.getAdapter(Message.class) : null;
	}

	/**
	 * Get the boolean flag to indicate if the editor input can be edited.
	 * 
	 * @return true if the input can be edited, false otherwise
	 */
	protected boolean isInputEditable()
	{
		boolean isEditable = true;

		if (this.input != null)
		{
			IEditableElement editableElement = (IEditableElement) this.input.getAdapter(IEditableElement.class);
			if (editableElement != null)
				isEditable = editableElement.canEdit();
		}

		return isEditable;
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/* Gets the default width to use for Text boxes displaying a class name */
	protected int getDefaultClassNameTextWidth()
	{
		return 380;
	}

	/* Creates a blank string for null values, and trims if required */
	protected String formatValue(String value, boolean trim)
	{
		String newValue = null;

		if (value == null)
			newValue = "";
		else if (trim)
			newValue = value.trim();
		else
			newValue = value;

		return newValue;
	}

	/* Get all available connection types or null if the operation fails */
	protected List<ConnectionType> getAvailableConnectionTypes()
	{
		List<ConnectionType> connectionTypeList = null;

		try
		{
			ServiceFactory serviceFactory = ServiceFactory.createFactory();
			IResourceTypeService service = serviceFactory.createResourceTypeService();
			connectionTypeList = service.getAllConnectionTypes();

			if (connectionTypeList == null)
				connectionTypeList = new ArrayList<ConnectionType>();
		}
		catch (Exception ex)
		{
			logger.error("Failed to get the list of available Connection Types.", ex);
		}

		return connectionTypeList;
	}
}
