package com.tandbergtv.watchpoint.studio.ui.model;

import static com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants.VARIABLE_SEID;

import java.util.ArrayList;
import java.util.List;

import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.ui.properties.DataType;

/**
 * Model representation of a WatchPoint Variable with data type.
 * 
 * @author Imran Naqvi
 * 
 */
public class WPVariable extends Variable implements IValidatableElement {

	String type;
	
	String value;
	
	private boolean compositeKey;

    private Object errors;

	private static final String START_TOKEN = "[";
	
	private static final String END_TOKEN = "]";
	
    private SuperStateVariableTransientInfo superStateInfo;

    /**
	 * @return the type
	 */
	public String getValue() {
		return value;
	}

	/**
	 * @param newType
	 *            the type to set
	 */
	public void setValue(String newValue) {
		String oldValue = this.value;
		this.value = newValue;
		firePropertyChange("value", oldValue, newValue);
	}
	
	/**
	 * @return the type
	 */
	public String getType() {
		return type;
	}

	/**
	 * @param newType
	 *            the type to set
	 */
	public void setType(String newType) {
		String oldType = this.type;
		this.type = newType;
		firePropertyChange("type", oldType, newType);
	}

	/**
	 * @return true if a Key, false otherwise
	 */
	public boolean isWOKey() {
		return this.getAccess() != null
				&& this.getAccess().indexOf(VariableAccessConstant.WOKEY_ACCESS) != -1;
	}

	/**
	 * @return true if a variable to display in UI, false otherwise
	 */
	public boolean isUIVariable() {
		return this.getAccess() != null
				&& this.getAccess().indexOf(VariableAccessConstant.SHOWINUI_ACCESS) != -1;
	}

	/**
	 * @return true if variable contains attachment name, false otherwise
	 */
	public boolean isAttachment() {
		return this.getAccess() != null
				&& this.getAccess().indexOf(VariableAccessConstant.ATTACHMENT_ACCESS) != -1;
	}

	/**
	 * @return the compositeKey
	 */
	public boolean isCompositeKey() {
		return compositeKey;
	}
	
	/**
	 * Returns true if the variable's name is a list expression of the form list_variable_name[index].
	 * Index must be numeric. list_variable_name must be of type LIST.
	 * 
	 * @return
	 */
	public boolean isListExpression() {
		return getName().contains(START_TOKEN) && getName().endsWith(END_TOKEN);
	}
	
	/**
	 * Returns the name of the list if this is a list expression. If the variable is files[i],
	 * files will be returned.
	 * 
	 * @return
	 */
	public String getListName() {
		if (isListExpression())
			return getName().substring(0, getName().indexOf(START_TOKEN));
		
		return null;
	}

	/**
	 * @param compositeKey the compositeKey to set
	 */
	public void setCompositeKey(boolean key) {
		this.compositeKey = key;
	}
	
	/**
	 * Creates a copy
	 * 
	 * @return
	 */
	public WPVariable cloneVariable() {
		WPVariable clone = (WPVariable) getFactory().createById(VARIABLE_SEID);
		
		clone.setAccess(getAccess());
		clone.setCompositeKey(compositeKey);
		clone.setName(getName());
		clone.setMappedName(getMappedName());
		clone.setType(this.type);
		clone.setValue(this.value);
		clone.setVariable(getVariable());
		clone.setFactory(this.getFactory());
		
		return clone;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		String tmp = getName();
		return (tmp == null || tmp.length() == 0) ? getMappedName() : tmp;
	}

	public boolean isNameValid() {
	    if (getName() != null) {
	        return getName().matches("[a-zA-Z_$][$\\w]*");
	    }
	    return false;
	}

    public void setReadable(boolean read) {
        setAccess(getAccesses(read, isWritable(), isRequired(), isAttachment()));
    }

    public void setWritable(boolean write) {
        setAccess(getAccesses(isReadable(), write, isRequired(), isAttachment()));
    }

    public void setRequired(boolean required) {
        setAccess(getAccesses(isReadable(), isWritable(), required, isAttachment()));
    }

    public void setAttachment(Boolean hasAttachment) {
        setAccess(getAccesses(isReadable(), isWritable(), isRequired(), hasAttachment));
    }

    private String getAccesses(boolean read, boolean write, boolean required, Boolean hasAttachment) {
        List<String> accesses = new ArrayList<String>();
        if (read) {
            accesses.add(VariableAccessConstant.READ_ACCESS);
        }
        if (write) {
            accesses.add(VariableAccessConstant.WRITE_ACCESS);
        }
        if (required) {
            accesses.add(VariableAccessConstant.REQUIRED_ACCESS);
        }
        if (hasAttachment) {
            accesses.add(VariableAccessConstant.ATTACHMENT_ACCESS);
        }

        StringBuilder result = new StringBuilder();
        for (String access : accesses) {
            result.append(access + ",");
        }
        if (accesses.size() > 0) {
            return result.substring(0, result.length() - 1);
        }
        return result.toString();
    }

    @Override
    public void setError(Object errors) {
        this.errors = errors;
    }

    @Override
    public Object getError() {
        return errors;
    }

    public void markError(Boolean value) {
        Object[] error = (Object[]) getError();
        if (error == null) {
            error = new Object[] { null, ""};
        }
        if (Boolean.TRUE.equals(value)) {
            setError(new Object[] { false, error[1] });
    
        } else {
            setError(new Object[] { !error[1].toString().isEmpty(), error[1] });
        }
    }

    public void addSuperStateInfo() {
        this.superStateInfo = new SuperStateVariableTransientInfo();
    }
    
    public SuperStateVariableTransientInfo getSuperStateInfo() {
        return superStateInfo;
    }

    /**
     * Type property was created when all variables were string. This is for backwards compatibility.
     */
    public String getEmptySafeType() {
        String type = getType();
        if (type == null || type.isEmpty()) {
            return DataType.STRING.toString();
        }
        return type;
    }
}