package com.tandbergtv.watchpoint.studio.ui.wizard;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.tandbergtv.watchpoint.studio.dto.ProtectionKey;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.util.UIValidator;

@Deprecated
public class ResourceGroupWizardPage extends WizardPage {

	public enum Mode {
		CREATE("Create Resource Group", "Create a new Resource Group"),
		VIEW("View Resource Group", null),
		EDIT("Edit Resource Group", null);
		
		private String title;
		private String description;

		Mode(String title, String description) {
			this.title = title;
			this.description = description;
		}
		
		public String getTitle() {
			return this.title;
		}
		
		public String getDescription() {
			return this.description;
		}
	}

	private Text nameText, allocationStrategyText, currentImageText, imageText;
	private Combo resourceTypeCombo, functionalTypeCombo, protectionKeyCombo;
	private Group allocationStyleGroup;
	private Button internalRadio, externalRadio, clearCurrentImageCheck, imageFileBrowseButton;
	private Map<String, ProtectionKey> protectionKeyNameObjectMap = new HashMap<String, ProtectionKey>();
	private Map<String, ResourceType> resourceTypeNameObjectMap = new HashMap<String, ResourceType>();

	private ModifyListener validationListener;
	
	private IResourceGroupService rgService;
	private IResourceTypeService rtService;
	private Mode mode;
	private ResourceGroup group;
	private ResourceType rtSelected;
	
	public ResourceGroupWizardPage(Mode mode, ResourceGroup group, ResourceType rtSelected) {
		super("Resource Group");
		this.mode = mode;
		setTitle(mode.getTitle());
		setDescription(mode.getDescription());
		this.rgService = ServiceFactory.createFactory().createResourceGroupService();
		this.rtService = ServiceFactory.createFactory().createResourceTypeService();
		if(mode != ResourceGroupWizardPage.Mode.CREATE)
			this.group = this.rgService.getResourceGroup(group.getKey());
		this.rtSelected = rtSelected;
		this.validationListener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				validate();
			}
		};
	}
	
	public boolean isPageComplete() {
		return (getErrorMessage() == null && nameText.getText().length() > 0);
	}
	
	//-----------------------------------------
	//		GETTER METHODS FOR PAGE VALUES
	//-----------------------------------------
	
	public String getResourceGroupName() {
		return nameText.getText();
	}
	
	public ResourceType getResourceType() {
		return resourceTypeNameObjectMap.get(resourceTypeCombo.getText());
	}
	
	public String getFunctionalType() {
		return functionalTypeCombo.getText();
	}

	/*
	 * @return true, if internal; false, if external
	 */
	public boolean isAllocationStyleInternal() {
		return internalRadio.getSelection();
	}
	
	public String getAllocationStrategy() {
		return allocationStrategyText.getText();
	}
	
	public boolean isClearCurrentImage() {
		return (this.mode != Mode.CREATE) ? clearCurrentImageCheck.getSelection() : false;
	}
	
	public String getImageFile() {
		return imageText.getText();
	}
	
	public ProtectionKey getProtectionKey() {
		String protectionKeyName = protectionKeyCombo.getText();
		return (protectionKeyNameObjectMap.containsKey(protectionKeyName))
				? protectionKeyNameObjectMap.get(protectionKeyName)
				: new ProtectionKey(protectionKeyName);
	}
		
	//-----------------------------------------
	//			CONTROL CREATION
	//-----------------------------------------

	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		Composite composite = createClientArea(parent);
		createNameControl(composite, null);
		createrResourceTypeControl(composite, nameText);
		createFunctionalTypeControl(composite, resourceTypeCombo);
		createAllocationStyleControl(composite, functionalTypeCombo);
		createAllocationStrategyControl(composite, allocationStyleGroup);
		if(this.mode == Mode.CREATE) {
			createImageControl(composite, allocationStrategyText);
		}
		else {
			createCurrentImageControl(composite, allocationStrategyText);
			createImageControl(composite, currentImageText);
		}
		createProtectionKeyCombo(composite, imageText);
		setValuesAndState();
		disableControls();
		setControl(composite);
		Dialog.applyDialogFont(composite);
		parent.getShell().pack();
	}

	private void disableControls() {
		switch(this.mode) {
		case CREATE:
			break;
		case EDIT:
			disable(nameText, resourceTypeCombo);
			break;
		case VIEW:
			disable(nameText, resourceTypeCombo, functionalTypeCombo,
					allocationStyleGroup, allocationStrategyText, clearCurrentImageCheck,
					imageText, imageFileBrowseButton, protectionKeyCombo);
		}
	}
	
	private void disable(Control... controls) {
		if(controls != null)
			for(Control control : controls)
				control.setEnabled(false);
	}
	
	private void setValuesAndState() {
		if(this.mode != ResourceGroupWizardPage.Mode.CREATE) {
			nameText.setText(convertNullToEmpty(group.getName()));
			
			if(group.getResourceType() != null)
				resourceTypeCombo.setText(convertNullToEmpty(group.getResourceType().getName()));
			
			functionalTypeCombo.setText(convertNullToEmpty(group.getFunctionalType()));
			
			internalRadio.setSelection(group.isInternallyAcquired());
			externalRadio.setSelection(!group.isInternallyAcquired());
			
			allocationStrategyText.setText(convertNullToEmpty(group.getAllocationStrategyClassName()));
			allocationStrategyText.setEnabled(group.isInternallyAcquired());
			
			currentImageText.setText(convertNullToEmpty(group.getImageFileName()));
			
			if(group.getImageContent() == null || group.getImageContent().length == 0)
				clearCurrentImageCheck.setEnabled(false);
			
			if(group.getProtectionKey() != null)
				protectionKeyCombo.setText(convertNullToEmpty(group.getProtectionKey().getName()));
		}else{
			if(this.rtSelected != null){
				resourceTypeCombo.setText(convertNullToEmpty(rtSelected.getName()));
				disable(resourceTypeCombo);
			}
		}
	}
	
	private String convertNullToEmpty(String str) {
		return ((str == null) ? "" : str);
	}
	
	/*
	 * Sets the layout of the client area.
	 * 
	 * @param parent
	 *            the parent composite of the client area
	 * @return the composite representing the client area
	 */
	private Composite createClientArea(Composite parent) {
		parent.getShell().setMinimumSize(400, 550);
		Composite composite = new Composite(parent, SWT.NONE);
		FormLayout layout = new FormLayout();
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		composite.setLayout(layout);
		return composite;
	}

	/*
	 * Creates the name text box along with its label.
	 * 
	 * @param parent
	 *            the parent composite of the text
	 */
	private void createNameControl(Composite parent, Control prevControl) {
		createLabel("Name : ", parent, prevControl);
		nameText = createText(parent, prevControl);
		nameText.setTextLimit(50);
		nameText.addModifyListener(validationListener);
	}
	
	/*
	 * Creates the node definition type combo box along with its label.
	 * 
	 * @param composite
	 *            the parent composite of the combo
	 */
	private void createrResourceTypeControl(Composite composite, Control prevControl) {
		createLabel("Resource Type : ", composite, prevControl);
		resourceTypeCombo = createCombo(composite, prevControl, true);
		fillResourceTypeCombo();
		resourceTypeCombo.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				String functionalTypeComboText = functionalTypeCombo.getText();
				populateFunctionalTypeCombo();
				functionalTypeCombo.setText(functionalTypeComboText);
			}
		});
	}
	
	/*
	 * Fills the node definition type combo. 
	 */
	private void fillResourceTypeCombo() {
		List<ResourceType> resourceTypes = this.rtService.getResourceTypeList();
		if(resourceTypes != null) {
			//sort resource types based on name (ignore case)
			Collections.sort(resourceTypes, new Comparator<ResourceType>() {
				public int compare(ResourceType o1, ResourceType o2) {
					return compareIgnoreCase(o1.getName(), o2.getName());
				}
			});
			for(ResourceType resourceType : resourceTypes) {
				resourceTypeCombo.add(resourceType.getName());
				resourceTypeNameObjectMap.put(resourceType.getName(), resourceType);
			}
			resourceTypeCombo.select(0);
		}
	}

	private void createFunctionalTypeControl(Composite parent, Control prevControl) {
		createLabel("Functional Type : ", parent, prevControl);
		functionalTypeCombo = createCombo(parent, prevControl, false);
		functionalTypeCombo.setTextLimit(256);
		populateFunctionalTypeCombo();
		functionalTypeCombo.addModifyListener(validationListener);
	}
	
	private void populateFunctionalTypeCombo() {
		ResourceType resourceType = getResourceType();
		if(resourceType == null)
			return;
		List<String> functionalTypes = rgService.getAllFunctionalTypesForResourceType(resourceType.getSystemId());
		if(functionalTypes != null) {
			//sort functional types (ignore case)
			Collections.sort(functionalTypes, new Comparator<String>() {
				public int compare(String o1, String o2) {
					return compareIgnoreCase(o1, o2);
				}
			});
			functionalTypeCombo.setItems(functionalTypes.toArray(new String[] {}));
		}
	}
	
	private void createAllocationStyleControl(Composite parent, Control prevControl) {
		createLabel("Allocation Style : ", parent, prevControl);
		allocationStyleGroup = createGroup(parent, prevControl);
		
		internalRadio = new Button(allocationStyleGroup, SWT.RADIO);
		internalRadio.setBounds(10, 15, 70, 15);
		internalRadio.setText("Internal");
		internalRadio.setSelection(true);
		SelectionListener sl = new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent event) {
				allocationStrategyText.setEnabled(internalRadio.getSelection());
				if(externalRadio.getSelection())
					allocationStrategyText.setText("");
				validate();
			}
		};
		internalRadio.addSelectionListener(sl);
		
		externalRadio = new Button(allocationStyleGroup, SWT.RADIO);
		externalRadio.setText("External");
		externalRadio.setBounds(120, 15, 70, 15);
		externalRadio.addSelectionListener(sl);
	}
	
	private void createAllocationStrategyControl(Composite parent, Control prevControl) {
		createLabel("Allocation Strategy: ", parent, prevControl);
		allocationStrategyText = createText(parent, prevControl);
		allocationStrategyText.setTextLimit(256);
		allocationStrategyText.addModifyListener(validationListener);
	}
	
	private void createCurrentImageControl(Composite parent, Control prevControl) {
		createLabel("Current Image: ", parent, prevControl);

		clearCurrentImageCheck = new Button(parent, SWT.CHECK);
		clearCurrentImageCheck.setText("Clear");
		FormData clearCurrentImageCheckFD = new FormData();
		clearCurrentImageCheckFD.right = new FormAttachment(100, -10);
		if(prevControl != null)
			clearCurrentImageCheckFD.top = new FormAttachment(prevControl, 10);
		clearCurrentImageCheck.setLayoutData(clearCurrentImageCheckFD);
		
		currentImageText = new Text(parent, SWT.BORDER);
		currentImageText.setEnabled(false);
		FormData currentImageTextFD = new FormData();
		currentImageTextFD.left = new FormAttachment(30, 2);
		currentImageTextFD.right = new FormAttachment(clearCurrentImageCheck, -10);
		if(prevControl != null)
			currentImageTextFD.top = new FormAttachment(prevControl, 10);
		currentImageText.setLayoutData(currentImageTextFD);
	}
	
	private void createImageControl(Composite parent, Control prevControl) {
		createLabel("Image : ", parent, prevControl);

		imageFileBrowseButton = new Button(parent, SWT.PUSH);
		imageFileBrowseButton.setText("Browse");
		FormData fd = new FormData();
		fd.right = new FormAttachment(100, -10);
		if(prevControl != null)
			fd.top = new FormAttachment(prevControl, 10);
		imageFileBrowseButton.setLayoutData(fd);
		imageFileBrowseButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent event) {
				String path = getPathFromUser();
				if(path != null)
					imageText.setText(path);
			}
		});

		imageText = new Text(parent, SWT.BORDER);
		imageText.addModifyListener(validationListener);
		fd = new FormData();
		fd.left = new FormAttachment(30, 2);
		fd.right = new FormAttachment(imageFileBrowseButton, -5);
		if(prevControl != null)
			fd.top = new FormAttachment(prevControl, 10);
		imageText.setLayoutData(fd);
	}
	
	private String getPathFromUser() {
		FileDialog fileDialog = new FileDialog(Display.getCurrent().getActiveShell(),
				SWT.SYSTEM_MODAL);
		fileDialog.setText("Please select the image file.");
		return fileDialog.open();
	}

	private void createProtectionKeyCombo(Composite parent, Control prevControl) {
		createLabel("Protection Key : ", parent, prevControl);
		protectionKeyCombo = createCombo(parent, prevControl, false);
		protectionKeyCombo.setTextLimit(50);
		protectionKeyCombo.addModifyListener(validationListener);
		protectionKeyCombo.addFocusListener(new FocusAdapter() {
			public void focusLost(FocusEvent e) {
					protectionKeyCombo.setText(protectionKeyCombo.getText().toUpperCase());				
			}
		});
		fillProtectionKeyCombo();
	}
	
	private void fillProtectionKeyCombo() {
		List<ProtectionKey> protectionKeys = rgService.getAllProtectionKeys();
		if(protectionKeys != null)
			//sort protection keys by their name (ignore case)
			Collections.sort(protectionKeys, new Comparator<ProtectionKey>() {
				public int compare(ProtectionKey o1, ProtectionKey o2) {
					return compareIgnoreCase(o1.getName(), o2.getName());
				}
			});
			for(ProtectionKey protectionKey : protectionKeys) {
				protectionKeyCombo.add(protectionKey.getName());
				protectionKeyNameObjectMap.put(protectionKey.getName(), protectionKey);
			}
	}
	
	//	-------------------------------------------------------------------------------
	//							GENERAL METHODS
	//	-------------------------------------------------------------------------------
	
	private void validate() {
		String errorMessage = null;
		
		errorMessage = UIValidator.validateValue("Name", nameText.getText(), UIValidator.Validation.NOT_BLANK, UIValidator.Validation.TRIMMED);
		if(errorMessage != null) {
			updateErrorMessage(errorMessage);
			return;
		}
		
		errorMessage = UIValidator.validateValue("Functional Type", functionalTypeCombo.getText(), UIValidator.Validation.TRIMMED);
		if(errorMessage != null) {
			updateErrorMessage(errorMessage);
			return;
		}
		
		if(internalRadio.getSelection()) {
			errorMessage = UIValidator.validateValue("Allocation Strategy", allocationStrategyText.getText(), UIValidator.Validation.TRIMMED);
			if(errorMessage != null) {
				updateErrorMessage(errorMessage);
				return;
			}
		}
		
		errorMessage = UIValidator.validateValue("Image", imageText.getText(), UIValidator.Validation.TRIMMED);
		if(errorMessage != null) {
			updateErrorMessage(errorMessage);
			return;
		}
		
		errorMessage = UIValidator.validateValue("Protection Key", protectionKeyCombo.getText(), UIValidator.Validation.NOT_BLANK, UIValidator.Validation.TRIMMED);
		if(errorMessage != null) {
			updateErrorMessage(errorMessage);
			return;
		}

		updateErrorMessage(null);
	}
	
	private void updateErrorMessage(String errorMessage) {
		setErrorMessage(errorMessage);
		getContainer().updateButtons();
	}
	
	private Label createLabel(String text, Composite parent, Control prevControl) {
		Label label = new Label(parent, SWT.NONE);
		label.setText(text);
		label.setLayoutData(getLSFormData(prevControl));
		return label;
	}
	
	private Text createText(Composite parent, Control prevControl) {
		Text text = new Text(parent, SWT.BORDER);
		text.setLayoutData(getRSFormData(prevControl));
		return text;
	}
	
	private Combo createCombo(Composite parent, Control prevControl, boolean readOnly) {
		int style = SWT.DROP_DOWN | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL;
		if(readOnly)
			style |= SWT.READ_ONLY;
		Combo combo = new Combo(parent, style);
		combo.setLayoutData(getRSFormData(prevControl));
		return combo;
	}
	
	private Group createGroup(Composite parent, Control prevControl) {
		Group group = new Group(parent, SWT.NULL);
		group.setLayoutData(getRSFormData(prevControl));
		return group;
	}
	
	private FormData getLSFormData(Control control) {
		FormData fd = new FormData();
		if(control != null)
			fd.top = new FormAttachment(control, 10);
		return fd;
	}
	
	private FormData getRSFormData(Control control) {
		FormData fd = new FormData();
		fd.left = new FormAttachment(30, 2);
		fd.right = new FormAttachment(100, -10);
		if(control != null)
			fd.top = new FormAttachment(control, 10);
		return fd;
	}
	
	private int compareIgnoreCase(String str1, String str2) {
		if(str1 == null && str2 == null)
			return 0;
		if(str1 == null)
			return -1;
		if(str2 == null)
			return 1;
		return str1.compareToIgnoreCase(str2);

	}
}
