/**
 * TemplateDataProvider.java
 * Created Feb 24, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.wizard.template;

import static javax.xml.xpath.XPathConstants.NODESET;

import java.io.StringReader;
import java.util.Collection;
import java.util.List;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.stream.StreamSource;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathException;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.jbpm.gd.jpdl.Plugin;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.external.fs.FileSystemContextFactory;
import com.tandbergtv.watchpoint.studio.external.fs.dao.DataAccessFactoryImpl;
import com.tandbergtv.watchpoint.studio.external.imports.ImportDataProvider;
import com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.service.impl.ServiceImpl;

/**
 * Provides a list of templates that can be imported into the persistent store
 * 
 * @author Sahil Verma
 */
public class TemplateDataProvider implements ImportDataProvider {

	private String location;

	private Collection<WorkflowTemplateDTO> existing;

	private Collection<ResourceGroup> groups;

	private Collection<NodeDefinitionDTO> nodes;

	private static final String NODE_DEFINITION_XPATH = "//nodeDefinition";

	private static final String RESOURCE_GROUP_XPATH = "//resource-group";

	private static final String RESOURCE_TYPE = "resourceType";

	private static final String DEFINITION_NAME = "definitionName";

	private static final String ID = "id";

	private static final org.apache.log4j.Logger logger = org.apache.log4j.Logger.getLogger(TemplateDataProvider.class);

	public TemplateDataProvider() {
		super();
	}

	/**
	 * {@inheritDoc}
	 */
	public String getImportLocation() {
		return this.location;
	}

	/**
	 * {@inheritDoc}
	 */
	public Collection<? extends IWatchPointDTO> getItems() {
		if (this.location == null)
			throw new IllegalArgumentException();
		
		logger.debug("Importing from " + location);

		this.existing = getExistingTemplates();
		ServiceFactory factory = ServiceFactory.createFactory();

		this.groups = factory.createResourceGroupService().getAllResourceGroups();
		this.nodes = factory.createNodeDefinitionService().getAllNodeDefinitions();
		
		ServiceImpl service = (ServiceImpl) ServiceFactory.createFactory().createWorkflowTemplateService();

		service.setDataAccessFactory(new DataAccessFactoryImpl());
		service.setPersistenceContextFactory(new FileSystemContextFactory(location));

		List<WorkflowTemplateDTO> templates = IWorkflowTemplateService.class.cast(service).getTemplateList();

		filter(templates);
		
		logger.debug("Found " + templates.size() + " items");

		return templates;
	}

	/**
	 * @param templates
	 */
	protected void filter(List<WorkflowTemplateDTO> templates) {
	}

	/**
	 * {@inheritDoc}
	 */
	public void setImportLocation(String location) {
		this.location = location;
	}

	protected Collection<WorkflowTemplateDTO> getExistingTemplates() {
		return ServiceFactory.createFactory().createWorkflowTemplateService().getTemplateList();
	}

	protected WorkflowTemplateDTO getExistingTemplate(String name) {
		for (WorkflowTemplateDTO template : this.existing) {
			if (name.equals(template.getName()))
				return template;
		}

		return null;
	}

	/**
	 * Ensures that the set of groups referred to by the template exist in the persistent store
	 * 
	 * @param template
	 * @return
	 */
	protected IStatus checkResourceGroups(WorkflowTemplateDTO template) {
		Element e = getDOM(template);
		XPath xpath = XPathFactory.newInstance().newXPath();
		String pluginId = Plugin.getDefault().getBundle().getSymbolicName();
		NodeList groups = null;
		
		try {
			groups = (NodeList) xpath.evaluate(RESOURCE_GROUP_XPATH, e, NODESET);
		} catch (XPathExpressionException e1) {
			throw new Error();
		}

		if (groups == null)
			return null;

		for (int i = 0; i < groups.getLength(); i++) {
			Element groupElement = (Element) groups.item(i);
			String name = groupElement.getAttribute("name");
			ResourceGroup group = null;

			/* Strictly speaking, we should examine the UID of the message(s) at this node in order
			 * to determine the systemId of the resource type, and ensure that the group is of that
			 * type */
			for (ResourceGroup rg : this.groups) {
				if (rg.getName().equals(name)) {
					group = rg;
					logger.debug(template.getName() + " - found group " + group.getName());
					break;
				}
			}

			if (group == null) {
				String message = template.getName() + " - no group named " + name + " found";
				return new Status(IStatus.ERROR, pluginId, message);
			}
		}
		
		return null;
	}

	/**
	 * Ensures that the node definitions referred to by the template exist in the persistent store.
	 * Side effect - if the node does exist, updates the ID in the DOM representation of the template.
	 * 
	 * @param template
	 * @return true if all nodes in the template exist in the persistent store, false otherwise
	 */
	protected IStatus checkNodeDefinitions(WorkflowTemplateDTO template) {
		Element e = getDOM(template);
		XPath xpath = XPathFactory.newInstance().newXPath();
		NodeList definitions = null;
		String pluginId = Plugin.getDefault().getBundle().getSymbolicName();
		
		try {
			definitions = (NodeList) xpath.evaluate(NODE_DEFINITION_XPATH, e, NODESET);
		} catch (XPathException e1) {
			throw new Error();
		}

		if (definitions == null)
			return null;

		for (int i = 0; i < definitions.getLength(); i++) {
			Element element = (Element) definitions.item(i);
			String name = element.getAttribute(DEFINITION_NAME);
			NodeDefinitionDTO node = null;

			/* FIXME Ensure that the node definition belongs to the right resource type.
			 * Also ensure the composite keys in the template match those in the node. */
			for (NodeDefinitionDTO nd : this.nodes) {
				if (nd.getName().equals(name)) {
					node = nd;

					/* Set the node definition ID and resource type ID in XML */
					element.setAttribute(ID, String.valueOf(nd.getId()));
					element.setAttribute(RESOURCE_TYPE, String.valueOf(nd.getResourceType().getId()));
					
					logger.debug(template.getName() + " - found node " + node.getName());

					break;
				}
			}

			if (node == null) {
				String message = template.getName() + " - no node definition named " + name + " found";
				return new Status(IStatus.ERROR, pluginId, message);
			}
		}
		
		return null;
	}

	protected Element getDOM(WorkflowTemplateDTO template) {
//		String xml = template.getXml();
		String xml = null;
		Element e = null;

		try {
			Document document = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
			TransformerFactory.newInstance().newTransformer().transform(
				new StreamSource(new StringReader(xml)), new DOMResult(document));

			e = document.getDocumentElement();
		} catch (Exception e1) {
			throw new Error();
		}

		return e;
	}
}
