/*
 * Created on Jul 11, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

import com.tandbergtv.watchpoint.studio.validation.IValidationRule;
import com.tandbergtv.watchpoint.studio.validation.IValidator;
import com.tandbergtv.watchpoint.studio.validation.ValidatorFactory;

/**
 * The default Validator Factory
 * 
 * @author Vijay Silva
 */
public class DefaultValidatorFactory extends ValidatorFactory
{
	// The logger
	private static final Logger logger = Logger.getLogger(DefaultValidatorFactory.class);

	// Map of validation rules
	private static final Map<Class<?>, List<IValidationRule<?>>> VALIDATION_RULE_MAP = new HashMap<Class<?>, List<IValidationRule<?>>>();

	// Flag to indicate if the validation rule map is initialized
	private static boolean MAP_INITIALIZED = false;

	/**
	 * Constructor
	 */
	public DefaultValidatorFactory()
	{
		super();

		initializeValidationRuleMap();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.ValidatorFactory#createValidator(java.lang.Class)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <T> IValidator<T> createValidator(Class<T> targetClass)
	{
		IValidator<T> validator = new RuleValidator<T>();

		if (VALIDATION_RULE_MAP.containsKey(targetClass))
		{
			List<?> rules = VALIDATION_RULE_MAP.get(targetClass);
			validator.setRules((List<IValidationRule<T>>) rules);
		}
		else
		{
			String msg = "Class: " + targetClass.getName()
					+ " is not specified in validation rule configuration,"
					+ " cannot create validator.";
			throw new ValidatorUndefinedException(msg);
		}

		return validator;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.ValidatorFactory#createValidator(java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <T> IValidator<T> createValidator(T target)
	{
		Class<T> targetClass = (Class<T>) target.getClass();
		return this.createValidator(targetClass);
	}

	/**
	 * The validation rules are read once and cached statically.
	 */
	private static synchronized void initializeValidationRuleMap()
	{
		if (MAP_INITIALIZED)
			return;

		try
		{
			VALIDATION_RULE_MAP.putAll(ValidationRuleReader.readValidationRules());
		}
		catch (ValidationRuleConfiguratonException vrce)
		{
			Map<Class<?>, List<IValidationRule<?>>> parsedRules = vrce.getValidationRules();
			if (parsedRules != null)
				VALIDATION_RULE_MAP.putAll(parsedRules);

			logger.error("Failed to load validation rule configuration.", vrce);
		}

		MAP_INITIALIZED = true;
	}
}
