/*
 * Created on Jul 18, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.template.graph;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.graph.ForkPathLabeler;
import com.tandbergtv.watchpoint.studio.validation.graph.GraphFactory;
import com.tandbergtv.watchpoint.studio.validation.graph.IWatchPointGraph;
import com.tandbergtv.watchpoint.studio.validation.graph.WatchPointDecorator;
import com.tandbergtv.watchpoint.studio.validation.graph.WatchPointGraphUtils;
import com.tandbergtv.watchpoint.studio.validation.graph.WorkflowTemplateGraph;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

import edu.uci.ics.jung.algorithms.connectivity.BFSDistanceLabeler;
import edu.uci.ics.jung.graph.Edge;
import edu.uci.ics.jung.graph.Vertex;
import edu.uci.ics.jung.utils.UserData;

/**
 * Performs the validation for each of the Fork-Join pairs present in the Workflow Template.
 * 
 * @author Vijay Silva
 */
public class TemplateForkJoinValidationRule extends ValidationRule<WorkflowTemplateGraph>
{
	private static final int MAX_ALLOWED_FORK_NEST_LEVEL = 1;

	private static final String FORK_VALID_KEY = "fork.isValid";

	/**
	 * Validates the Fork-Join pairs
	 * 
	 * @param graph
	 *            The Workflow Template Graph to validate
	 * 
	 * @return The list of validation messages returned
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(WorkflowTemplateGraph graph)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		/* Evaluate all Fork Paths */
		WorkflowTemplateGraph graphCopy = (WorkflowTemplateGraph) graph.copy();
		ForkPathLabeler forkLabeler = new ForkPathLabeler();
		forkLabeler.labelForkPaths(graphCopy);

		/* Decorator used to mark fork nodes as valid or invalid */
		WatchPointDecorator<String, Boolean> validityDecorator = new WatchPointDecorator<String, Boolean>(
				FORK_VALID_KEY, UserData.SHARED);

		/* Stage 1 Validation: validate the number of fork and join nodes */
		this.validateForkJoinCount(forkLabeler, messages);

		/* Stage 2 Validation: find invalid nested forks and forks with many / missing joins */
		Set<Vertex> forkVertices = forkLabeler.getForkVertices();
		for (Vertex forkVertex : forkVertices)
		{
			validityDecorator.setValue(true, forkVertex);

			// Validate the nested level for every fork node
			this.validateForkNestLevel(forkVertex, forkLabeler, messages, validityDecorator);

			// Validate the fork goes to one common join
			this.validateForkToSingleJoin(forkVertex, forkLabeler, messages, validityDecorator);
		}

		/* Stage 3 Validation: find joins with many / missing forks */
		Set<Vertex> joinVertices = forkLabeler.getJoinVertices();
		for (Vertex joinVertex : joinVertices)
		{
			this.validateJoinFromSingleFork(joinVertex, forkLabeler, messages, validityDecorator);
		}

		/*
		 * Stage 4 Validation: validate that fork - join reachable vertices are not directly
		 * reachable from other nodes and that nodes are not shared by transitions of a Fork.
		 */
		for (Vertex forkVertex : forkVertices)
		{
			// Skip invalid fork nodes
			if (!validityDecorator.getValue(forkVertex))
				continue;

			// Get the Join Vertex (there should only be one join vertex)
			Vertex joinVertex = this.getJoinVertex(forkVertex, forkLabeler);

			// Build the list of vertices that are part of the 'fork' sub-graph
			Set<Vertex> vertices = this.getForkGraphVertices(forkVertex, forkLabeler);

			// Validate that all nodes in the Fork / Join pairing can reach the Join node
			this.validateForkJoinReachability(forkVertex, joinVertex, vertices, forkLabeler,
					messages, validityDecorator);

			if (!validityDecorator.getValue(forkVertex))
				continue;

			// Validate that the Fork Subgraph is disjoint
			this.validateForkSubgraphDisjoint(forkVertex, joinVertex, vertices, forkLabeler,
					messages, validityDecorator);

			if (!validityDecorator.getValue(forkVertex))
				continue;

			// Validate that each path from each transition of a Fork is dijoint
			this.validateForkTransitionsDisjoint(forkVertex, forkLabeler, messages,
					validityDecorator);
		}

		return messages;
	}

	// ========================================================================
	// ==================== VALIDATION METHODS
	// ========================================================================

	/*
	 * Validates that the Fork and Join node count is equal
	 */
	private void validateForkJoinCount(ForkPathLabeler labeler, List<ValidationMessage> messages)
	{
		int forkCount = labeler.getForkVertices().size();
		int joinCount = labeler.getJoinVertices().size();

		if (forkCount != joinCount)
		{
			Object element = WatchPointGraphUtils.getElement(labeler.getGraph());
			ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.TEMPLATE_FORK_JOIN_UNEQUAL);
		}
	}

	/*
	 * Validates that the Fork node is not nested more than it should be
	 */
	private void validateForkNestLevel(Vertex forkVertex, ForkPathLabeler labeler,
			List<ValidationMessage> messages, WatchPointDecorator<String, Boolean> decorator)
	{
		int nestLevel = labeler.getForkNestLevel(forkVertex);
		if (nestLevel > MAX_ALLOWED_FORK_NEST_LEVEL)
		{
			Object element = WatchPointGraphUtils.getElement(forkVertex);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.INVALID_FORK_NEST_LEVEL);
			decorator.setValue(false, forkVertex);
		}
	}

	/*
	 * Validate that the fork node transitions all reach a single join transition
	 */
	private void validateForkToSingleJoin(Vertex forkVertex, ForkPathLabeler forkLabeler,
			List<ValidationMessage> messages, WatchPointDecorator<String, Boolean> decorator)
	{
		Set<Vertex> joinVertices = forkLabeler.getJoinVertices(forkVertex);
		int joinVerticesCount = (joinVertices != null) ? joinVertices.size() : 0;
		if (joinVerticesCount != 1)
		{
			Object element = WatchPointGraphUtils.getElement(forkVertex);
			ValidationMessageCode code = (joinVerticesCount == 0)
					? ValidationMessageCode.FORK_MISSING_JOIN_PAIR
					: ValidationMessageCode.FORK_MAPPED_TO_MANY_JOINS;
			ValidationMessageAdder.getInstance().addValidationMessage(messages, element, code);
			decorator.setValue(false, forkVertex);
		}
		else
		{ // Check for fork transitions with no joins
			for (Object edgeObject : forkVertex.getOutEdges())
			{
				if (forkLabeler.getJoinVertices((Edge) edgeObject).size() == 0)
				{
					Object element = WatchPointGraphUtils.getElement(forkVertex);
					ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.FORK_TRANSITION_MISSING_JOIN_PAIR);
					decorator.setValue(false, forkVertex);
					break;
				}
			}
		}
	}

	/*
	 * Validate that a join node is reached by only one Fork
	 */
	private void validateJoinFromSingleFork(Vertex joinVertex, ForkPathLabeler forkLabeler,
			List<ValidationMessage> messages, WatchPointDecorator<String, Boolean> decorator)
	{
		Set<Vertex> forkVertices = forkLabeler.getForkVertices(joinVertex);
		int forkVerticesCount = (forkVertices != null) ? forkVertices.size() : 0;
		if (forkVerticesCount != 1)
		{
			Object element = WatchPointGraphUtils.getElement(joinVertex);
			ValidationMessageCode code = (forkVerticesCount == 0)
					? ValidationMessageCode.JOIN_MISSING_FORK_PAIR
					: ValidationMessageCode.JOIN_MAPPED_TO_MANY_FORKS;
			ValidationMessageAdder.getInstance().addValidationMessage(messages, element, code);

			if (forkVerticesCount > 1)
			{
				for (Vertex forkVertex : forkVertices)
				{ // Mark the fork vertices as invalid
					decorator.setValue(false, forkVertex);
				}
			}
		}
	}

	/*
	 * Validate the paths from the Fork node, ensuring that every node reachable from the Fork node
	 * has a path to the corresponding Join Node. Also validates that every node in the Fork
	 * subgraph (other than the Fork Node) is not reachable from outside the subgraph.
	 */
	private void validateForkJoinReachability(Vertex forkVertex, Vertex joinVertex,
			Set<Vertex> forkGraphVertices, ForkPathLabeler labeler,
			List<ValidationMessage> messages, WatchPointDecorator<String, Boolean> decorator)
	{
		// Build a 'reversed' fork sub-graph.
		IWatchPointGraph graph = labeler.getGraph();
		IWatchPointGraph graphCopy = GraphFactory.createFactory().createReverseGraph(graph);
		for (Object vertexObject : graph.getVertices())
		{
			Vertex vertex = (Vertex) vertexObject;
			if (!forkGraphVertices.contains(vertex))
			{
				Vertex equalVertex = (Vertex) vertex.getEqualVertex(graphCopy);
				graphCopy.removeVertex(equalVertex);
			}
		}

		// Verify that every node in the reversed fork subgraph is reachable from the Join
		BFSDistanceLabeler distanceLabeler = new BFSDistanceLabeler("fork.subgraph.distance");
		distanceLabeler.labelDistances(graphCopy, (Vertex) joinVertex.getEqualVertex(graphCopy));
		if (distanceLabeler.getUnivistedVertices().size() > 0)
		{
			Object element = WatchPointGraphUtils.getElement(forkVertex);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.FORK_PATH_MISSING_JOIN);
			decorator.setValue(false, forkVertex);
		}
	}

	/*
	 * Validates that the Fork Subgraph is disjoint from the rest of the graph. Every vertex in the
	 * fork subgraph should not be reachable from the rest of the graph, except through the fork
	 * node.
	 */
	private void validateForkSubgraphDisjoint(Vertex forkVertex, Vertex joinVertex,
			Set<Vertex> forkGraphVertices, ForkPathLabeler labeler,
			List<ValidationMessage> messages, WatchPointDecorator<String, Boolean> decorator)
	{
		boolean isValid = true;
		for (Vertex vertex : forkGraphVertices)
		{
			boolean isFork = vertex.equals(forkVertex);
			boolean isJoin = vertex.equals(joinVertex);

			for (Object vertexObject : vertex.getPredecessors())
			{
				Vertex predecessor = (Vertex) vertexObject;
				boolean isInSubgraph = forkGraphVertices.contains(predecessor);

				/* Dealing with the Fork node's predecessor */
				if (isFork)
				{
					if (isInSubgraph && !(predecessor.equals(joinVertex)))
					{ // The predecessor is not Join, and Fork reached from within the subgraph
						Object element = WatchPointGraphUtils.getElement(forkVertex);
						ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.FORK_REACHED_FROM_FORK_SUBGRAPH);
						isValid = false;
						break;
					}
				}

				/* Dealing with the Join node's predecessor */
				else if (isJoin)
				{
					if (predecessor.equals(forkVertex))
					{ // Predecessor of Join is Fork
						Object element = WatchPointGraphUtils.getElement(forkVertex);
						ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.FORK_JOIN_PATH_EMPTY);
						isValid = false;
						break;
					}
					else if (!isInSubgraph)
					{ // Predecessor of Join not in Fork subgraph
						Object element = WatchPointGraphUtils.getElement(joinVertex);
						ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.JOIN_REACHED_OUTSIDE_FORK_SUBGRAPH);
						isValid = false;
						break;
					}
				}

				/* Dealing with the Fork sub-graph node's predecessor */
				else if (!isInSubgraph)
				{ // Predecessor of fork subgraph node not in the subgraph
					Object element = WatchPointGraphUtils.getElement(forkVertex);
					ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.FORK_SUBGRAPH_REACHED_OUTSIDE_FORK_SUBGRAPH);
					isValid = false;
					break;
				}
			}

			if (!isValid)
			{
				decorator.setValue(false, forkVertex);
				break;
			}
		}
	}

	/*
	 * Validates that all transitions in the Fork - Join group do not visit any common nodes.
	 */
	private void validateForkTransitionsDisjoint(Vertex forkVertex, ForkPathLabeler labeler,
			List<ValidationMessage> messages, WatchPointDecorator<String, Boolean> decorator)
	{
		Set<Vertex> reachableVertices = new HashSet<Vertex>();
		boolean isValid = true;

		for (Object edgeObject : forkVertex.getOutEdges())
		{
			Edge forkEdge = (Edge) edgeObject;
			Set<Vertex> edgeReachableVertices = labeler.getReachableVertices(forkEdge);
			if (reachableVertices.size() == 0)
			{ // Initialize the reachable vertices
				reachableVertices.addAll(edgeReachableVertices);
			}
			else
			{
				// Check if any vertices overlap
				for (Vertex vertex : edgeReachableVertices)
				{
					if (reachableVertices.contains(vertex))
					{
						Object element = WatchPointGraphUtils.getElement(forkVertex);
						ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.FORK_TRANSITIONS_OVERLAP_NODES);
						isValid = false;
						break;
					}

					reachableVertices.add(vertex);
				}
			}

			if (!isValid)
			{ // Set the validity flag for fork node
				decorator.setValue(false, forkVertex);
				break;
			}
		}
	}

	// ========================================================================
	// ==================== HELPER METHODS
	// ========================================================================

	/*
	 * Get the Join Vertex for the given fork vertex and the labeler that maintains the join
	 * vertices for each of the fork edges.
	 */
	private Vertex getJoinVertex(Vertex forkVertex, ForkPathLabeler labeler)
	{
		Vertex joinVertex = null;

		Set<Vertex> joinVertices = labeler.getJoinVertices(forkVertex);
		if (joinVertices != null && joinVertices.size() > 0)
		{
			joinVertex = joinVertices.iterator().next();
		}

		return joinVertex;
	}

	/*
	 * Builds a set of vertices that are reachable from any path from the Fork Vertex and ending at
	 * the corresponding join vertex.
	 */
	private Set<Vertex> getForkGraphVertices(Vertex forkVertex, ForkPathLabeler labeler)
	{
		Set<Vertex> forkGraphVertices = new HashSet<Vertex>();

		forkGraphVertices.add(forkVertex);
		forkGraphVertices.addAll(labeler.getJoinVertices(forkVertex));
		forkGraphVertices.addAll(labeler.getReachableVertices(forkVertex));

		return forkGraphVertices;
	}
}
