/*
 * Created on Jul 17, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.template.graph;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.jbpm.gd.jpdl.model.EndState;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.StartState;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.graph.WatchPointGraphUtils;
import com.tandbergtv.watchpoint.studio.validation.graph.WorkflowTemplateGraph;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;
import com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph.NodeReachableRule;

import edu.uci.ics.jung.graph.Vertex;

/**
 * Validation rule that ensures that every node in the Workflow Template is 'reachable'. A node is
 * reachable if and only if a path exists from the 'Start' node to that node, and a path exists from
 * that node to the 'End' node.
 * 
 * @author Vijay Silva
 */
public class TemplateNodeReachableRule extends NodeReachableRule<WorkflowTemplateGraph>
{
	/**
	 * Get the Start Vertex.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph.NodeReachableRule#getRootVertices(com.tandbergtv.watchpoint.studio.validation.graph.IWatchPointGraph)
	 */
	@Override
	protected Set<Vertex> getRootVertices(WorkflowTemplateGraph graph)
	{
		Set<Vertex> startVertices = new HashSet<Vertex>();

		/* Find the start node vertices */
		for (Object vertexObject : graph.getVertices())
		{
			Vertex vertex = (Vertex) vertexObject;
			NodeElement node = WatchPointGraphUtils.getWTVertexElement(vertex);
			if (node instanceof StartState)
			{
				startVertices.add(vertex);
			}
		}

		return startVertices;
	}

	/**
	 * Get the End Vertex.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph.NodeReachableRule#getSinkVertices(com.tandbergtv.watchpoint.studio.validation.graph.IWatchPointGraph)
	 */
	@Override
	protected Set<Vertex> getSinkVertices(WorkflowTemplateGraph graph)
	{
		Set<Vertex> endVertices = new HashSet<Vertex>();

		/* Find the end node vertices */
		for (Object vertexObject : graph.getVertices())
		{
			Vertex vertex = (Vertex) vertexObject;
			NodeElement node = WatchPointGraphUtils.getWTVertexElement(vertex);
			if (node instanceof EndState)
			{
				endVertices.add(vertex);
			}
		}

		return endVertices;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph.NodeReachableRule#validateNodesUnreachableFromRoot(java.util.Set,
	 *      java.util.List)
	 */
	@Override
	protected void validateNodesUnreachableFromRoot(Set<Vertex> vertices,
			final List<ValidationMessage> messages)
	{
		this.validateUnreachableNode(vertices, messages, true);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelementcontainer.graph.NodeReachableRule#validateNodesUnreachableToSink(java.util.Set,
	 *      java.util.List)
	 */
	@Override
	protected void validateNodesUnreachableToSink(Set<Vertex> vertices,
			final List<ValidationMessage> messages)
	{
		this.validateUnreachableNode(vertices, messages, false);
	}

	/*
	 * Create the validation messages for all nodes not reachable from the Start Node
	 */
	private void validateUnreachableNode(Set<Vertex> vertices, List<ValidationMessage> messages,
			boolean fromStart)
	{
		if (vertices == null)
			return;

		for (Vertex vertex : vertices)
		{
			NodeElement node = WatchPointGraphUtils.getWTVertexElement(vertex);
			ValidationMessageCode code = (fromStart)
					? ValidationMessageCode.NODE_UNREACHABLE_FROM_START
					: ValidationMessageCode.NODE_UNREACHABLE_TO_END;
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code);
		}
	}
}
