package com.tandbergtv.watchpoint.studio.debugger.ui;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.forms.widgets.FormToolkit;

import com.tandbergtv.watchpoint.studio.debugger.WatchpointTemplateDebuggerPlugin;
import com.tandbergtv.watchpoint.studio.debugger.launching.TemplateLauncherConstants;
import com.tandbergtv.watchpoint.studio.debugger.model.NodeSimulationConfig;
import com.tandbergtv.watchpoint.studio.debugger.util.DebuggerUtils;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * 		The TAB for the NodeSimulation 
 * 
 * @author <a href="mailto:francisco.bento.silva.neto@ericsson.com">efrasio - Francisco Bento da Silva Neto</a>
 *
 */
public class NodeSimulatorTab extends AbstractLaunchConfigurationTab implements SelectionListener {

	AbstractNodeSimulationComposite nodeSimulationComposite;
	NodeSimulationTable templateNodesTable;
	NodeSimulationConfig currentNode = null;
	Map<String, NodeSimulationConfig> simulationConfigMap = new HashMap<String, NodeSimulationConfig>();
	
	Group nodeSimulationGroup;
	
	Composite mainPanel = null;
	FormToolkit toolkit = null;
	
	protected String templatePath = EMPTY_STRING; 
	protected WorkflowTemplate template; 
	
	public static final String EMPTY_STRING = "";
	
	public NodeSimulatorTab() {
	}
	
	//-----------------------------------------
	//			CONTROL CREATION
	//-----------------------------------------

	protected void createMainPanel(Composite parent) {
    	mainPanel = new Composite(parent, SWT.NONE);
    	mainPanel.setLayout(new GridLayout(1, false));
    	mainPanel.setFont(parent.getFont());
    	
    	GridData gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 1;
		mainPanel.setLayoutData(gd);
	}
	
	protected void createTemplateNodesGroup() {
		Group grpTemplateNodes = new Group(mainPanel, SWT.NONE);
		grpTemplateNodes.setLayout(new FillLayout(SWT.HORIZONTAL));
		grpTemplateNodes.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1));
		grpTemplateNodes.setText("Template Nodes");
		toolkit.paintBordersFor(grpTemplateNodes);
		templateNodesTable = new NodeSimulationTable(grpTemplateNodes);
		templateNodesTable.addSelectionListener(this);
		templateNodesTable.addSimulationModifyListener(new SimulationModifyListener() {
			public void modifySimulation(Object o) {
				updateLaunchConfigurationDialog();
				if (nodeSimulationComposite != null) {
					nodeSimulationComposite.load(templateNodesTable.getSelection());
				}
			}

		});
	}
	
	protected void createNodeSimulationGroup() {
		nodeSimulationGroup = new Group(mainPanel, SWT.NONE);
		nodeSimulationGroup.setBackground(mainPanel.getBackground());
		nodeSimulationGroup.setLayout(new FillLayout(SWT.HORIZONTAL));
		GridData gridData = new GridData(SWT.FILL, SWT.TOP, true, false, 1, 1);
		gridData.heightHint = 290;
		nodeSimulationGroup.setLayoutData(gridData);
		nodeSimulationGroup.setText("Node Simulation Properties");
		toolkit.paintBordersFor(nodeSimulationGroup);
	}
	
	protected void createSeparator() {
		Label label = new Label(mainPanel, SWT.SEPARATOR | SWT.HORIZONTAL);
		label.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1));
	}

	@SuppressWarnings("unchecked")
	public void updateTemplate(String templatePath, ILaunchConfiguration config) {
		template = DebuggerUtils.parseWorkspaceTemplate(templatePath);
		
		Collection<NodeSimulationConfig> simulationData = Collections.EMPTY_LIST;
		if (template != null) {
			try {
				simulationConfigMap = NodeSimulationConfigFactory.createFactory().getNodesForSimulation(template, config.getAttributes());
				simulationData = simulationConfigMap.values();
			} catch (CoreException e) {
			}
		} 
		templateNodesTable.setSimulationData(simulationData);
	}
	
	public void createControl(final Composite parent) {
		toolkit = new FormToolkit(parent.getDisplay());

		createMainPanel(parent);
		createTemplateNodesGroup();
		createSeparator();
		createNodeSimulationGroup();
		
		setControl(mainPanel);
	}
	
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
	}

	public void initializeFrom(ILaunchConfiguration config) {
		try {
			String newTemplatePath = config.getAttribute(TemplateLauncherConstants.TEMPLATE_LAUNCHER_PERSISTENT_DATA_TEMPLATE, EMPTY_STRING);
			// Template has changed, reset the UI and the template data 
			if (!newTemplatePath.equals(templatePath)) {
				if (nodeSimulationComposite != null) {
					nodeSimulationComposite.dispose(); 
				}
				templatePath = newTemplatePath;
				updateTemplate(templatePath, config);
			}
			
		} catch (CoreException e) {
		}
	}

	@SuppressWarnings("rawtypes")
	public void performApply(ILaunchConfigurationWorkingCopy config) {
		for (NodeSimulationConfig nodeConfig : simulationConfigMap.values()) {
			Map nodeAttributes = nodeConfig.getSimulationData().getAttributes();
			
			String nodeName = nodeConfig.getNodeElement().getName();
			if (nodeConfig.getSimulationType() != null) {
				config.setAttribute(TemplateLauncherConstants.getNodeSimulationTypeKey(nodeName), nodeConfig.getSimulationType().getValue());
			} else {
				config.removeAttribute(TemplateLauncherConstants.getNodeSimulationTypeKey(nodeName));
			}
			String dataPrefix = TemplateLauncherConstants.getNodeSimulationDataKey(nodeName);
			for (Object keyObj : nodeAttributes.keySet()) {
				String key = keyObj.toString();
				
				String newKey = dataPrefix + "." + key;
				config.setAttribute(newKey, nodeAttributes.get(keyObj).toString());				
			}

		}
	}

	@Override
	public Image getImage() {
		return Utility.getImageDescriptor(WatchpointTemplateDebuggerPlugin.PLUGIN_ID, "template_simulation_tab.png").createImage();
	}

	public String getName() {
		return "Node Simulation";
	}

	public void widgetSelected(SelectionEvent e) {
		NodeSimulationConfig selected = templateNodesTable.getSelection();
		
		boolean isSameNode = false;
		if (selected != null) {
			if (currentNode != null) {
				if (currentNode.getNodeElement().getName().equals(selected.getNodeElement().getName())) {
					// Same node, do nothing.
					isSameNode = true;
				}
			}
		}

		if (!isSameNode) {
			if (nodeSimulationComposite != null) {
				nodeSimulationComposite.dispose(); 
			}
			// Searches the factory for the Composite responsible for editing the Node specific data
			nodeSimulationComposite = 
				NodeSimulationWidgetFactory.getCompositeForNode(templateNodesTable.getSelection(), 
																nodeSimulationGroup);
			nodeSimulationComposite.addSimulationModifyListener(new SimulationModifyListener() {
				public void modifySimulation(Object o) {
					updateLaunchConfigurationDialog();
					templateNodesTable.updateNode((NodeSimulationConfig) o);
				}
			});
            nodeSimulationComposite.setTemplate(template);
			nodeSimulationComposite.load(templateNodesTable.getSelection());
			setControl(mainPanel);
			
			// Hack for adjusting toolbars/resizing  
			ScrolledComposite sc = (ScrolledComposite) mainPanel.getParent();
			sc.setMinSize(mainPanel.computeSize(SWT.DEFAULT, SWT.DEFAULT));
			
			super.updateLaunchConfigurationDialog();
		}
		
		currentNode = selected;
	}

	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}

}
