package com.tandbergtv.watchpoint.studio.builder;

import static com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO.GPD_FILE_EXTENSION;
import static com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO.TEMPLATE_DEFINITION_FILE_EXTENSION_NAME;

import java.util.ArrayList;
import java.util.List;

import javax.xml.transform.TransformerException;
import javax.xml.xpath.XPathException;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.jbpm.gd.common.model.NamedElement;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.builder.resolution.template.WatchpointTemplateResolutionGenerator;
import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.external.fs.dao.TemplateParser;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.external.wpexport.IWatchPointDTOExporter;
import com.tandbergtv.watchpoint.studio.external.wpexport.WatchPointDTOExporterFactory;
import com.tandbergtv.watchpoint.studio.external.wpexport.impl.WorkflowTemplateDocumentExporter;
import com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.view.DTONameDecorator;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;

public class WatchpointTemplateVisitor implements IResourceDeltaVisitor, IResourceVisitor {
    // The Logger
	private static final Logger logger = Logger.getLogger(WatchpointTemplateVisitor.class);

	private static final String MARKER_TYPE = "org.jbpm.gd.jpdl.templateProblem";
	
	/**
	 * This method is invoked when a resource changes.
	 * 
	 * @see org.eclipse.core.resources.IResourceDeltaVisitor#visit(org.eclipse.core.resources.IResourceDelta)
	 */
	public boolean visit(IResourceDelta delta) throws CoreException {
		boolean visitChilden = true;
		
		IResource resource = delta.getResource();
		
		// Verify if its not the "classes" or "bin" folder where the java compiler creates the compiled files.
		if (!isClassesFolder(resource)) {
			
			boolean isTemplate = checkProcessDefinitioXML(resource);
			boolean isTemplateGpd = checkTemplateGpdXML(resource);
			
			if (isTemplate) {
				switch (delta.getKind()) {
				case IResourceDelta.ADDED:
				case IResourceDelta.CHANGED:
					// handle added or changed resource
					saveOrUpdate(resource);
					break;
				case IResourceDelta.REMOVED:
					// handle removed resource
					deleteTemplate(resource);
					break;
				}
			} else {
				// Also need to generate template document when gpd file is modified.
				if (isTemplateGpd){
					IFile jpdlFile = getJpdlFileForGpd(resource);
					try {
						WorkflowTemplateDocumentExporter.getInstance().export(jpdlFile);
					} catch (Exception e) {
						throw new RuntimeException("Error export template document for: " + resource, e);

					}
				}
				ResourceGroupVisitor visitor = new ResourceGroupVisitor();
				visitor.visit(delta);
			}
		} else {
			// Its the "bin" or "classes" folder, skip visiting
			visitChilden = false;
		}
		
		return visitChilden;
	}

	/**
	 * Invoked when a FULL BUILD is executed.
	 * 
	 * @see org.eclipse.core.resources.IResourceVisitor#visit(org.eclipse.core.resources.IResource)
	 */
	public boolean visit(IResource resource) {
		boolean visitChilden = true;
		
		// Verify if its not the "classes" or "bin" folder where the java compiler put the compiled files.
		if (!isClassesFolder(resource)) {
			boolean isTemplate = checkProcessDefinitioXML(resource);
			
			if (isTemplate) {
				saveOrUpdate(resource);
			} else {
				ResourceGroupVisitor visitor = new ResourceGroupVisitor();
				visitor.visit(resource);
			}
		} else {
			// Its the "bin" or "classes" folder, skip visiting
			visitChilden = false;
		}
		
		//return true to continue visiting children.
		return visitChilden;
	}
	
	/**
	 * 		Validates the template, and apply the markers when needed.
	 * @param file
	 * @param template
	 */
	void validateTemplate(IFile file, WorkflowTemplateDTO template) {
		IWorkflowTemplateService service = ServiceFactory.createFactory().createWorkflowTemplateService();
		
		List<ValidationMessage> messages = service.validateTemplate(template);
		try {
			file.deleteMarkers(MARKER_TYPE, false, IResource.DEPTH_ZERO);
		} catch (CoreException e) {
			logger.error("Error deleting markers", e);
		}
		boolean hasErrors = false;
		boolean hasWarnings = false;
		if (messages != null && !messages.isEmpty()) {
			for (ValidationMessage message : messages) {
				try {
					IMarker marker = file.createMarker(MARKER_TYPE);
					String msg = message.getCode();
					try {
						msg = com.tandbergtv.watchpoint.studio.ui.util.ValidationMessages.getInstance().getMessage(message);
						String elementName = "";
						if (message.getElement() instanceof NamedElement) {
							elementName = ((NamedElement) message.getElement()).getName();
						} else if (message.getElement() instanceof IWatchPointDTO) {
							elementName = new DTONameDecorator((IWatchPointDTO)message.getElement()).getName();
						}
						
						if (elementName != null && !elementName.isEmpty()) {
							msg = elementName + " : " + msg;
						}
					} catch (Exception e) {
						logger.error("Error reading properties", e);
					} 
					marker.setAttribute(IMarker.MESSAGE, msg);
					
					// defaul severity
					int severity = IMarker.SEVERITY_ERROR;
					switch (message.getType()) {
						case Error:
							hasErrors = true;
							break;
					
						case Warning:
							hasWarnings = true;
							severity = IMarker.SEVERITY_WARNING;
							break;
	
						case Notification:
							severity = IMarker.SEVERITY_INFO;
							break;
					}
					marker.setAttribute(IMarker.SEVERITY, severity);
					marker.setAttribute(IMarker.LINE_NUMBER, 1);
					marker.setAttribute(WatchpointTemplateResolutionGenerator.ERROR_CODE, message.getCode());
					if (message.getElement() instanceof NodeDefinition) {
						NodeDefinition node = (NodeDefinition) message.getElement();
						marker.setAttribute(WatchpointTemplateResolutionGenerator.ERROR_ELEMENT, node.getName());						
					}
					if(message.getElement() instanceof Variable){
						Variable variable = (Variable) message.getElement();
						marker.setAttribute(WatchpointTemplateResolutionGenerator.ERROR_ELEMENT, variable.getName());
					}

                    message.copyAttributesTo(marker);

				} catch (CoreException e) {
					logger.error("Error adding marker", e);
				}
			}
		} 
		template.setHasErrors(hasErrors);
		template.setHasWarnings(hasWarnings);
		service.saveTemplate(template);
	}
	
	
	/**
	 * 		Saves or update the template represented by the resource.
	 * 
	 * @param resource
	 */
	void saveOrUpdate(IResource resource) {
		IFile file = (IFile) resource;
		try {
			WorkflowTemplateDTO template = TemplateParser.parse(file);
			template.setProjectName(file.getProject().getName());
			template.setPath(resource.getFullPath().toPortableString());
			
			IWorkflowTemplateService service = ServiceFactory.createFactory().createWorkflowTemplateService();

			WorkflowTemplateDTO previousTemplate = service.getTemplateByPath(template.getPath());
			String previousTemplateName = null; 
			if (previousTemplate != null) {
				previousTemplateName = previousTemplate.getName();
			}
			
			service.saveTemplate(template);
			
			validateTemplate(file, template);
			validateNameChanges(previousTemplateName, template);
			validateReferencingTemplates(template);
			
			// Use the Export Service to export the Template
			WatchPointDTOExporterFactory exporterFactory = WatchPointDTOExporterFactory.createFactory();
			IWatchPointDTOExporter<WorkflowTemplateDTO> exporter = exporterFactory.getExporter(WorkflowTemplateDTO.class);
			exporter.export(template, null);
			
			// When template jpdl file is changed, need to generate the workflow document.
			IFile templateJpdlFile = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(template.getPath()));
			WorkflowTemplateDocumentExporter.getInstance().export(templateJpdlFile);
			
			org.jbpm.gd.jpdl.Logger.logInfo("Finished building template " + template.getName());
		} catch (TransformerException e) {
			org.jbpm.gd.jpdl.Logger.logError("Error building template " + file.getLocation(), e);
			logger.error("Error parsing template " + file.getLocation(), e);
		} catch (XPathException e) {
			org.jbpm.gd.jpdl.Logger.logError("Error building template " + file.getLocation(), e);
			logger.error("Error parsing template " + file.getLocation(), e);
		} catch (CoreException e) {
			org.jbpm.gd.jpdl.Logger.logError("Error building template " + file.getLocation(), e);
			logger.error("Error parsing template " + file.getLocation(), e);
		} catch (ExportFailureException e) {
			org.jbpm.gd.jpdl.Logger.logError("Error building template " + file.getLocation(), e);
			logger.error("Error exporting template " + file.getLocation(), e);
		} catch (Exception e) {
			org.jbpm.gd.jpdl.Logger.logError("Error building template " + file.getLocation(), e);
			logger.error("Error building template " + file.getLocation(), e);
		}
	}
	
	/**
	 * 		Re-Validate possible templates associated with the current one.
	 * If this template is referenced by another template through the use of the SubProcess feature,
	 * this gonna cause all the referencing templates to be re-validated.  
	 *  
	 * @param template The template changed
	 */
	void validateReferencingTemplates(WorkflowTemplateDTO template) {
		IWorkflowTemplateService service = ServiceFactory.createFactory().createWorkflowTemplateService();
		List<WorkflowTemplateDTO> referencingTemplates = service.getReferencingTemplates(template.getName());
		
		for (WorkflowTemplateDTO referencingTemplate : referencingTemplates) {
			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(referencingTemplate.getPath()));
			validateTemplate(file, referencingTemplate);
		}
	}
	
	/**
	 * 		Validate name changes. 
	 * 		1. If the template name has changed, have to check if its matching
	 * another template name and then add validation errors (markers) to these templates.
	 * 		2. If the template name has changed and its not matching another template name,
	 * have to clear the possibly previous errors (markers).
	 *  
	 * @param previousTemplateName
	 * @param template
	 */
	void validateNameChanges(String previousTemplateName, WorkflowTemplateDTO template) {
		IWorkflowTemplateService service = ServiceFactory.createFactory().createWorkflowTemplateService();
		if (previousTemplateName != null) {
			// If name has changed, call builder for the templates with the old name
			if (!previousTemplateName.equals(template.getName())) {
				List<WorkflowTemplateDTO> previousNameTemplates = service.getTemplateByName(previousTemplateName);
				for (WorkflowTemplateDTO previousNameTemplate : previousNameTemplates) {
					if (!previousNameTemplate.getPath().equals(template.getPath())) {
						IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(previousNameTemplate.getPath()));
						validateTemplate(file, previousNameTemplate);
					}
				}
	 		}
		} 
		
		// If name has changed, call builder for the templates with the new name
		List<WorkflowTemplateDTO> newNameTemplates = service.getTemplateByName(template.getName());
		for (WorkflowTemplateDTO newNameTemplate : newNameTemplates) {
			if (!newNameTemplate.getPath().equals(template.getPath())) {
				IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(newNameTemplate.getPath()));
				validateTemplate(file, newNameTemplate);
			}
		}
	}
	
	/**
	 * 		Deletes the template.
	 * 
	 * @param resource
	 */
	public void deleteTemplate(IResource resource) {
		IWorkflowTemplateService service = ServiceFactory.createFactory().createWorkflowTemplateService();
		WorkflowTemplateDTO template = service.getTemplateByPath(resource.getFullPath().toPortableString());
		if (template != null) {
			delete(template);
		}
	}
	
	/**
	 * 		Deletes all templates from a specified project.
	 * 
	 * @param resource
	 */
	public void delete(IProject resource) {
		IWorkflowTemplateService service = ServiceFactory.createFactory().createWorkflowTemplateService();
		
		List<WorkflowTemplateDTO> templates = service.getTemplatesByProject(resource.getName());
		
		/* Deletes the Templates on two-steps: 
		   . First the parent templates, then the child templates:
		   		Otherwise the child Templates will cause parent templates to be re-validated
		   		which doesn't makes sense, because parent templates will be deleted as well.  */
		List<WorkflowTemplateDTO> secondBatchTemplates = new ArrayList<WorkflowTemplateDTO>();
		for (WorkflowTemplateDTO template : templates) {
			if (!template.getSubprocesses().isEmpty()) {
				delete(template);	
			} else {
				secondBatchTemplates.add(template);
			}
		}
		for (WorkflowTemplateDTO template : secondBatchTemplates) {
			delete(template);	
		}
	}
	
	/**
	 * 		Deletes the Template and re-validate all depending templates.
	 * 
	 * @param deletedTemplate
	 */
	protected void delete(WorkflowTemplateDTO deletedTemplate) {
		IWorkflowTemplateService service = ServiceFactory.createFactory().createWorkflowTemplateService();
		service.deleteTemplate(deletedTemplate.getPath());
		
		/* Re-Validate templates with the same name */ 
		List<WorkflowTemplateDTO> templates = service.getTemplateByName(deletedTemplate.getName());
		for (WorkflowTemplateDTO template : templates) {
			if (!deletedTemplate.getPath().equals(template.getPath())) {
				IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(template.getPath()));
				validateTemplate(file, template);
			}
		}
		
		validateReferencingTemplates(deletedTemplate);
	}
	
	/**
	 * 		Check if its a valid template file.
	 * @param resource
	 * @return
	 */
	private boolean checkProcessDefinitioXML(IResource resource) {
		boolean isProcessDefinition = false;
		
		if (resource instanceof IFile && TEMPLATE_DEFINITION_FILE_EXTENSION_NAME.equals(resource.getFileExtension())) {
			isProcessDefinition = true;
		}
		return isProcessDefinition;
	}
	
	/**
	 * 		Check if its a valid gpd file.
	 * @param resource
	 * @return
	 */
	private boolean checkTemplateGpdXML(IResource resource) {
		if (resource instanceof IFile && GPD_FILE_EXTENSION.equalsIgnoreCase("." + resource.getFileExtension())){
			IFile jpdlFile = getJpdlFileForGpd(resource);
			if (jpdlFile.exists()){
				return true;
			}
		}
		return false;
	}
	
	private IFile getJpdlFileForGpd(IResource resource){
		IContainer parentFolder = ((IFile) resource).getParent();
		return parentFolder.getFile(new Path(parentFolder.getName() + "." + WorkflowTemplateDTO.TEMPLATE_DEFINITION_FILE_EXTENSION_NAME));
	}

	/**
	 * 		Check if its the java compiled classes folder - usually project/bin or project/classes . 
	 * 
	 * @param resource
	 * @return
	 */
	private boolean isClassesFolder(IResource resource) {
		boolean isClassesFolder = false;
		
		IJavaProject project = JavaCore.create(resource.getProject());
		try {
			IPath outputFolder = project.getOutputLocation();
			if (outputFolder.isPrefixOf(resource.getFullPath())) {
				isClassesFolder = true;
			}
		} catch (JavaModelException e) {
			// ignore as its just for optimization
		}
		return isClassesFolder;
	}
	
}
