/*
 * Created on Jun 5, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dto;

import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * The ResourceType Entity
 * 
 * @author Vijay Silva
 */
public class ResourceType implements IWatchPointDTO, IVersionable, IPersistable
{
	private static final long serialVersionUID = -1762516537061906221L;

	private long id = DEFAULT_ID;
	
	private String path;
	
	private String projectName;

	private String systemId;

	private String name;

	private String description;

	private Date createDate;

	private ConnectionType connectionType;

	private int messageSequenceNumber = 0;

	private AdaptorType adaptorType;

	private String adaptorClassName;

	private AdaptorScope adaptorScope;

	private String messageIdentificationClassName;

	private boolean initializationRequired = false;

	private String initializationStrategyClass;

	private boolean heartbeatRequired = false;

	private String heartbeatStrategyClass;

	private ConnectionType heartbeatConnectionType;

	private Set<ResourceTypeParameter> parameters;

	private String defaultProtocol;

	private Set<Message> messages;
	
	private Set<NodeDefinitionDTO> nodes;

	private int version;

	private boolean dirty;

	private boolean outOfTheBox;
	
	private boolean hasErrors;
	
	private boolean hasWarnings;

	/**
	 * Default Constructor
	 */
	public ResourceType()
	{
		super();
		this.nodes = new HashSet<NodeDefinitionDTO>();
		this.messages = new HashSet<Message>();
	}

	/**
	 * Creates a Resource Type with the specified Id.
	 * 
	 * @param id
	 *            The Resource Type Id
	 */
	public ResourceType(long id)
	{
		this.setId(id);
	}

	/**
	 * Constructor used when creating Resource Type View Items.
	 * 
	 * @param id
	 *            The Resource Type Id
	 * @param name
	 *            The Name
	 * @param systemId
	 *            The System Id
	 * @param description
	 *            The Description
	 * @param createDate
	 *            The Create Date
	 * @param version
	 *            The Version Number
	 * @param dirty
	 *            Flag to indicate if the Resource Type is dirty
	 * @param outOfTheBox
	 *            Flag to indicate if the Resource Type is out-of-the-box
	 */
	public ResourceType(long id, String name, String systemId, String description, Date createDate,
			int version, boolean dirty, boolean outOfTheBox)
	{
		this();
		this.setId(id);
		this.setName(name);
		this.setSystemId(systemId);
		this.setDescription(description);
		this.setConnectionType(connectionType);
		this.setCreateDate(createDate);
		this.setVersion(version);
		this.setDirty(dirty);
		this.setOutOfTheBox(outOfTheBox);
	}

	/**
	 * Gets the ResourceType Id
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dto.IPersistable#getKey()
	 */
	public Long getKey()
	{
		return this.id;
	}

	/**
	 * @return the id
	 */
	public long getId()
	{
		return this.id;
	}

	/**
	 * @param id
	 *            the id to set
	 */
	public void setId(long id)
	{
		this.id = id;
	}

	/**
	 * Sets the path of the plugin.xml file
	 * 
	 * @param path
	 */
	public void setPath(String path) {
		this.path = path;
	}

	/**
	 * Gets the path of the plugin.xml file
	 * @return
	 */
	public String getPath() {
		return path;
	}

	public void setProjectName(String projectName) {
		this.projectName = projectName;
	}

	public String getProjectName() {
		return projectName;
	}

	/**
	 * @return the systemId
	 */
	public String getSystemId()
	{
		return this.systemId;
	}

	/**
	 * @param systemId
	 *            the systemId to set
	 */
	public void setSystemId(String systemId)
	{
		this.systemId = systemId;
	}

	/**
	 * @return the name
	 */
	public String getName()
	{
		return this.name;
	}

	/**
	 * @param name
	 *            the name to set
	 */
	public void setName(String name)
	{
		this.name = name;
	}

	/**
	 * @return the description
	 */
	public String getDescription()
	{
		return this.description;
	}

	/**
	 * @param description
	 *            the description to set
	 */
	public void setDescription(String description)
	{
		this.description = description;
	}
	
	/*
	 * To handle the case when the eclipse project name is not the same as the resource type name
	 * Specify the eclipse project name at the beginning of the "description" field, with "[","]" surrounded.
	 * e.g. <parameter id="description" value="[azuki] Resource Type for Azuki device."/>
	 * "azuki" is the eclipse project name, while the resource type name is "Azuki"
	 */
	public String determineNameForProject(){
		String projectName = getProjectName();
		if (projectName == null || projectName.trim().isEmpty()){
			projectName = getName();
		}
//		
//		String description = getDescription();
//		if (description != null && !description.trim().isEmpty()){
//			description = description.trim();
//			if (description.indexOf("[") == 0){
//				int endIndex = description.indexOf("]");
//				if (endIndex != -1){
//					projectName = description.substring(1, endIndex);
//				}
//			}
//		}
		return projectName;
	}

	/**
	 * @return the createDate
	 */
	public Date getCreateDate()
	{
		return this.createDate;
	}

	/**
	 * @param createDate
	 *            the createDate to set
	 */
	public void setCreateDate(Date createDate)
	{
		this.createDate = createDate;
	}

	/**
	 * @return the connectionType
	 */
	public ConnectionType getConnectionType()
	{
		return this.connectionType;
	}

	/**
	 * @param connectionType
	 *            the connectionType to set
	 */
	public void setConnectionType(ConnectionType connectionType)
	{
		this.connectionType = connectionType;
	}

	/**
	 * The Sequence number used for generating Message UIDs
	 * 
	 * @return the Message Sequence Number
	 */
	public int getMessageSequenceNumber()
	{
		return this.messageSequenceNumber;
	}

	/**
	 * @param messageSequenceNumber
	 *            the Message Sequence Number to set
	 */
	public void setMessageSequenceNumber(int messageSequenceNumber)
	{
		this.messageSequenceNumber = messageSequenceNumber;
	}

	/**
	 * @return the adaptorType
	 */
	public AdaptorType getAdaptorType()
	{
		if (this.adaptorType == null)
			this.adaptorType = AdaptorType.NONE;

		return this.adaptorType;
	}

	/**
	 * @param adaptorType
	 *            the adaptorType to set
	 */
	public void setAdaptorType(AdaptorType adaptorType)
	{
		this.adaptorType = adaptorType;
	}

	/**
	 * @return the adaptorClassName
	 */
	public String getAdaptorClassName()
	{
		return this.adaptorClassName;
	}

	/**
	 * @param adaptorClassName
	 *            the adaptorClassName to set
	 */
	public void setAdaptorClassName(String adaptorClassName)
	{
		this.adaptorClassName = adaptorClassName;
	}

	/**
	 * Get all the Parameters used in the configuration of the Resource Type. Changing the contents
	 * of the returned set does not affect the state of the Resource Type.
	 * 
	 * @return A Set of all the configuration parameters.
	 */
	public Set<ResourceTypeParameter> getAllParameters()
	{
		Set<ResourceTypeParameter> allParameters = new HashSet<ResourceTypeParameter>();

		if (parameters != null)
			allParameters.addAll(this.parameters);

		return allParameters;
	}

	/**
	 * @return the parameters
	 */
	public Set<ResourceTypeParameter> getParameters()
	{
		return this.parameters;
	}

	/**
	 * @param parameters
	 *            the parameters to set
	 */
	protected void setParameters(Set<ResourceTypeParameter> parameters)
	{
		this.parameters = parameters;
	}

	/**
	 * @return the scope
	 */
	public AdaptorScope getAdaptorScope()
	{
		return this.adaptorScope;
	}

	/**
	 * @param scope
	 *            the scope to set
	 */
	public void setAdaptorScope(AdaptorScope scope)
	{
		this.adaptorScope = scope;
	}

	/**
	 * @return the messageIdentificationClassName
	 */
	public String getMessageIdentificationClassName()
	{
		return this.messageIdentificationClassName;
	}

	/**
	 * @param messageIdentificationClassName
	 *            the messageIdentificationClassName to set
	 */
	public void setMessageIdentificationClassName(String messageIdentificationClassName)
	{
		this.messageIdentificationClassName = messageIdentificationClassName;
	}

	/**
	 * @return the heartbeatConnectionType
	 */
	public ConnectionType getHeartbeatConnectionType()
	{
		return this.heartbeatConnectionType;
	}

	/**
	 * @param heartbeatConnectionType
	 *            the heartbeatConnectionType to set
	 */
	public void setHeartbeatConnectionType(ConnectionType heartbeatConnectionType)
	{
		this.heartbeatConnectionType = heartbeatConnectionType;
	}

	/**
	 * @return the heartbeatStrategyClass
	 */
	public String getHeartbeatStrategyClass()
	{
		return this.heartbeatStrategyClass;
	}

	/**
	 * @param heartbeatStrategyClass
	 *            the heartbeatStrategyClass to set
	 */
	public void setHeartbeatStrategyClass(String heartbeatStrategyClass)
	{
		this.heartbeatStrategyClass = heartbeatStrategyClass;
	}

	/**
	 * @return the initializationStrategyClass
	 */
	public String getInitializationStrategyClass()
	{
		return this.initializationStrategyClass;
	}

	/**
	 * @param initializationStrategyClass
	 *            the initializationStrategyClass to set
	 */
	public void setInitializationStrategyClass(String initializationStrategyClass)
	{
		this.initializationStrategyClass = initializationStrategyClass;
	}

	/**
	 * @return the requiresHeartbeat
	 */
	public boolean isHeartbeatRequired()
	{
		return this.heartbeatRequired;
	}

	/**
	 * @param requiresHeartbeat
	 *            the requiresHeartbeat to set
	 */
	public void setHeartbeatRequired(boolean requiresHeartbeat)
	{
		this.heartbeatRequired = requiresHeartbeat;
	}

	/**
	 * @return the requiresInitialization
	 */
	public boolean isInitializationRequired()
	{
		return this.initializationRequired;
	}

	/**
	 * @param requiresInitialization
	 *            the requiresInitialization to set
	 */
	public void setInitializationRequired(boolean requiresInitialization)
	{
		this.initializationRequired = requiresInitialization;
	}

	/**
	 * @return the defaultProtocol
	 */
	public String getDefaultProtocol()
	{
		return this.defaultProtocol;
	}

	/**
	 * @param defaultProtocol
	 *            the defaultProtocol to set
	 */
	public void setDefaultProtocol(String defaultProtocol)
	{
		this.defaultProtocol = defaultProtocol;
	}

	/**
	 * @return the messages
	 */
	public Set<Message> getMessages()
	{
		return this.messages;
	}
	
	/**
	 * Returns a map of messages, keyed by uid
	 * 
	 * @return
	 */
	public Map<String, Message> getMessagesMap()
	{
		Map<String, Message> map = new HashMap<String, Message>();
		
		for (Message message : this.messages)
			map.put(message.getUid(), message);
		
		return map;
	}

	/**
	 * @param messages
	 *            the messages to set
	 */
	public void setMessages(Set<Message> messages)
	{
		this.messages = messages;
	}

	/**
	 * @return the nodes
	 */
	public Set<NodeDefinitionDTO> getNodes()
	{
		return nodes;
	}

	/**
	 * @param nodes the nodes to set
	 */
	protected void setNodes(Set<NodeDefinitionDTO> nodes)
	{
		this.nodes = nodes;
	}
	
	/**
	 * Adds a node definition
	 * 
	 * @param node
	 */
	public void addNode(NodeDefinitionDTO node)
	{
		this.nodes.add(node);
		node.setResourceType(this);
	}
	
	/**
	 * Returns the node that has the given name, if such a node exists
	 * 
	 * @param name
	 */
	public NodeDefinitionDTO getNode(String name)
	{
		for (NodeDefinitionDTO node : this.nodes)
		{
			if (node.getName().equals(name))
				return node;
		}
		
		return null;
	}
	
	/**
	 * Removes the specified node
	 * 
	 * @param node
	 */
	public void removeNode(NodeDefinitionDTO node)
	{
		this.nodes.remove(node);
	}

	/**
	 * @return the version
	 */
	public int getVersion()
	{
		return this.version;
	}

	/**
	 * @param version
	 *            the version to set
	 */
	public void setVersion(int version)
	{
		this.version = version;
	}

	/**
	 * @return the isDirty
	 */
	public boolean isDirty()
	{
		return this.dirty;
	}

	/**
	 * @param dirty
	 *            the isDirty to set
	 */
	public void setDirty(boolean dirty)
	{
		this.dirty = dirty;
	}
	
	public boolean isHasErrors() {
		return hasErrors;
	}

	public void setHasErrors(boolean hasErrors) {
		this.hasErrors = hasErrors;
	}

	public boolean isHasWarnings() {
		return hasWarnings;
	}

	public void setHasWarnings(boolean hasWarnings) {
		this.hasWarnings = hasWarnings;
	}

	/**
	 * Add a new Message to the list of Messages for the Resource Type
	 * 
	 * @param message
	 *            The Message to add
	 */
	public void addMessage(Message message)
	{
		message.setResourceType(this);
		messages.add(message);
	}

	/**
	 * Removes a Message from the list of Messages for the Resource Type. Does nothing is the
	 * message is not part of the Resource Type messages.
	 * 
	 * @param message
	 *            The Message to remove
	 */
	public void removeMessage(Message message)
	{
		if (messages == null)
			return;
		
		for (Message messageRT : messages) {
			if(messageRT.getName().equals(message.getName())){
				messages.remove(messageRT);
				break;
			}
		}

		messages.remove(message);
	}
	
	/**
	 * Gets a message using the given uid
	 * 
	 * @param uid
	 * @return
	 */
	public Message getMessage(String uid)
	{
		if (messages == null)
			return null;
		
		for (Message message : messages)
		{
			if (message.getUid().equals(uid))
				return message;
		}
		
		return null;
	}

	/**
	 * @return the outOfTheBox
	 */
	public boolean isOutOfTheBox()
	{
		return outOfTheBox;
	}

	/**
	 * @param outOfTheBox
	 *            the outOfTheBox to set
	 */
	public void setOutOfTheBox(boolean outOfTheBox)
	{
		this.outOfTheBox = outOfTheBox;
	}

	// ==============================================================
	// ============= ADAPTOR PARAMETER MANAGEMENT
	// ==============================================================

	/**
	 * Gets the map of Adaptor Parameters. Modifying the returned map does not affect the
	 * ResourceType object.
	 * 
	 * @return the Adaptor Parameters
	 */
	public Map<String, String> getAdaptorParameters()
	{
		return this.getParameters(ParameterType.ADAPTOR);
	}

	/**
	 * Sets the list of Adaptor Parameters, removing all currently existing parameters
	 * 
	 * @param adaptorParameters
	 *            The map of adaptor parameters
	 */
	public void setAdaptorParameters(Map<String, String> adaptorParameters)
	{
		this.setParameters(ParameterType.ADAPTOR, adaptorParameters);
	}

	/**
	 * Add a new Adaptor Parameter to the list of existing parameters. Both the name and value
	 * cannot be null.
	 * 
	 * @param name
	 *            The Adaptor Parameter name
	 * @param value
	 *            The Value
	 */
	public void addAdaptorParameter(String name, String value)
	{
		this.addParameter(ParameterType.ADAPTOR, name, value);
	}

	/**
	 * Removes the Adaptor Parameter
	 * 
	 * @param name
	 *            the Adaptor Parameter name
	 * @return The value associated with the parameter
	 */
	public String removeAdaptorParameter(String name)
	{
		return this.removeParameter(ParameterType.ADAPTOR, name);
	}

	/**
	 * Remove all the Adaptor Parameters.
	 */
	public void removeAdaptorParameters()
	{
		this.removeParameters(ParameterType.ADAPTOR);
	}

	// ==============================================================
	// ============= HEARTBEAT PARAMETER MANAGEMENT
	// ==============================================================

	/**
	 * Gets the map of HeartBeat parameters. Modifying the returned map does not affect the
	 * ResourceType object.
	 * 
	 * @return the HeartBeat Parameters
	 */
	public Map<String, String> getHeartbeatParameters()
	{
		return this.getParameters(ParameterType.HEARTBEAT);
	}

	/**
	 * Sets the map of HeartBeat parameters.
	 * 
	 * @param heartbeatParameters
	 *            The HeartBeat parameters
	 */
	public void setHeartbeatParameters(Map<String, String> heartbeatParameters)
	{
		this.setParameters(ParameterType.HEARTBEAT, heartbeatParameters);
	}

	/**
	 * Add a new HeartBeat Parameter to the list of existing parameters
	 * 
	 * @param name
	 *            The HeartBeat Parameter name
	 * @param value
	 *            The Value
	 */
	public void addHeartbeatParameter(String name, String value)
	{
		this.addParameter(ParameterType.HEARTBEAT, name, value);
	}

	/**
	 * Removes the HeartBeat parameter, returning the value associated it.
	 * 
	 * @param name
	 *            The Parameter Name
	 * @return The Parameter Value, or null if the name doesn't exist.
	 */
	public String removeHearbeatParameter(String name)
	{
		return this.removeParameter(ParameterType.HEARTBEAT, name);
	}

	/**
	 * Remove all the HeartBeat Parameters.
	 */
	public void removeHeartbeatParameters()
	{
		this.removeParameters(ParameterType.HEARTBEAT);
	}

	// ==============================================================
	// ============= INITIALIZATION PARAMETER MANAGEMENT
	// ==============================================================

	/**
	 * Gets the map of Initialization parameters. Modifying the returned map does not affect the
	 * ResourceType object.
	 * 
	 * @return the Initialization Parameters
	 */
	public Map<String, String> getInitializationParameters()
	{
		return this.getParameters(ParameterType.INITIALIZATION);
	}

	/**
	 * Sets the map of Initialization parameters.
	 * 
	 * @param parameters
	 *            The Initialization parameters
	 */
	public void setInitializationParameters(Map<String, String> parameters)
	{
		this.setParameters(ParameterType.INITIALIZATION, parameters);
	}

	/**
	 * Add a new Initialization Parameter to the list of existing parameters
	 * 
	 * @param name
	 *            The Initialization Parameter name
	 * @param value
	 *            The Value
	 */
	public void addInitializationParameter(String name, String value)
	{
		this.addParameter(ParameterType.INITIALIZATION, name, value);
	}

	/**
	 * Remove the Initialization Parameter, returning the currently associated value (or null if no
	 * such parameter exists).
	 * 
	 * @param name
	 *            The Initialization Parameter Name
	 */
	public void removeInitializationParameter(String name)
	{
		this.removeParameter(ParameterType.INITIALIZATION, name);
	}

	/**
	 * Remove all the Initialization Parameters.
	 */
	public void removeInitializationParameters()
	{
		this.removeParameters(ParameterType.INITIALIZATION);
	}

	// ========================================================================
	// ================= PARAMETER MANAGEMENT
	// ========================================================================

	public boolean hasParameter(ParameterType type, ResourceTypeParameter parameter)
	{
		for (ResourceTypeParameter p : this.parameters)
		{
			if (isMatchingParameter(p, type, parameter.getName()))
				return true;
		}
		
		return false;
	}
	
	/**
	 * Get the map of key-value pairs for a parameter type
	 * 
	 * @param parameterType
	 *            The Parameter Type
	 * @return The map of configuration values.
	 */
	protected Map<String, String> getParameters(ParameterType parameterType)
	{
		Map<String, String> result = new HashMap<String, String>();

		if (this.parameters == null)
			return result;

		for (ResourceTypeParameter parameter : this.parameters)
		{
			if (parameter.getType() == parameterType)
			{
				result.put(parameter.getName(), parameter.getValue());
			}
		}

		return result;
	}

	/**
	 * Set the Parameters for a particular Parameter Type
	 * 
	 * @param parameterType
	 *            The Parameter Type
	 * @param newParameters
	 *            The Map of new name - values for the parameter type
	 */
	protected void setParameters(ParameterType parameterType, Map<String, String> newParameters)
	{
		this.removeParameters(parameterType);
		if (newParameters == null)
			return;

		for (String name : newParameters.keySet())
			this.addParameter(parameterType, name, newParameters.get(name));
	}

	/**
	 * Remove all the parameters of the specified type.
	 * 
	 * @param parameterType
	 *            The Parameter Type
	 */
	protected void removeParameters(ParameterType parameterType)
	{
		if (this.parameters == null)
			return;

		for (Iterator<ResourceTypeParameter> itr = this.parameters.iterator(); itr.hasNext();)
		{
			ResourceTypeParameter parameter = itr.next();
			if (parameter.getType() == parameterType)
			{
				itr.remove();
			}
		}
	}

	/**
	 * Add a configuration parameter, or updates the value of an existing parameter if the name and
	 * type matches.
	 * 
	 * @param name
	 *            The Parameter Name
	 * @param parameterType
	 *            The Parameter Type
	 * @param value
	 *            the Parameter Value
	 */
	public void addParameter(ParameterType parameterType, String name, String value)
	{
		if (this.parameters == null)
			this.parameters = new HashSet<ResourceTypeParameter>();

		if (name == null || parameterType == null || value == null)
		{
			String msg = "ResourceType Parameter type, name and value must all be not null.";
			throw new IllegalArgumentException(msg);
		}

		/* Try and update the existing parameter if present */
		for (ResourceTypeParameter parameter : this.parameters)
		{
			if (isMatchingParameter(parameter, parameterType, name))
			{
				parameter.setValue(value);
				return;
			}
		}

		/* Add new parameter since it doesn't exist */
		this.parameters.add(new ResourceTypeParameter(this, parameterType, name, value));
	}

	/**
	 * Remove the Parameter with the specified name and type
	 * 
	 * @param name
	 *            The Parameter Name
	 * @param parameterType
	 *            The Parameter Type
	 * @return The value of the parameter being removed, or null if it does not exists.
	 */
	protected String removeParameter(ParameterType parameterType, String name)
	{
		String oldValue = null;

		if (this.parameters == null)
			return oldValue;

		for (Iterator<ResourceTypeParameter> itr = this.parameters.iterator(); itr.hasNext();)
		{
			ResourceTypeParameter parameter = itr.next();
			if (this.isMatchingParameter(itr.next(), parameterType, name))
			{
				itr.remove();
				oldValue = parameter.getValue();
				break;
			}
		}
		return oldValue;
	}

	/**
	 * Gets the value associated with a parameter
	 * 
	 * @param name
	 *            The Parameter name
	 * @param parameterType
	 *            The Parameter Type
	 * @return The Parameter value, or null if it doesn't exist
	 */
	protected String getParameterValue(ParameterType parameterType, String name)
	{
		String value = null;

		if (this.parameters == null)
			return value;

		for (ResourceTypeParameter parameter : this.parameters)
		{
			if (isMatchingParameter(parameter, parameterType, name))
			{
				value = parameter.getValue();
				break;
			}
		}
		return value;
	}

	/*
	 * Check if the parameter matches the name and type specified.
	 */
	private boolean isMatchingParameter(ResourceTypeParameter parameter, ParameterType type,
			String name)
	{
		String parameterName = parameter.getName();
		ParameterType parameterType = parameter.getType();
		return ((parameterType == type) && parameterName.equals(name));
	}

	// ========================================================================
	// ================= OVERRIDEN METHODS
	// ========================================================================

	/**
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode()
	{
		if (this.id == DEFAULT_ID)
			return super.hashCode();

		return new Long(this.id).hashCode() * 3 + 484;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj)
	{
		if (!(obj instanceof ResourceType))
			return false;

		ResourceType resourceType = (ResourceType) obj;

		/* Check for the Default Id */
		if (this.id == DEFAULT_ID && resourceType.id == DEFAULT_ID)
			return super.equals(resourceType);

		return (this.id == resourceType.id);
	}

	/**
	 * @see java.lang.Object#clone()
	 */
	@Override
	public ResourceType clone()
	{
		ResourceType clone = null;

		try
		{
			clone = (ResourceType) super.clone();

			/* Clone the Parameters */
			if (this.parameters != null)
			{
				clone.parameters = new HashSet<ResourceTypeParameter>();
				for (ResourceTypeParameter parameter : this.parameters)
				{
					clone.parameters.add(parameter.clone());
				}
			}

			/* Clone the Messages */
			if (this.messages != null)
			{
				clone.messages = new HashSet<Message>();
				for (Message message : this.messages)
				{
					clone.addMessage(message.clone());
				}
			}

			/* Clone the connection type */
			if (this.connectionType != null)
				clone.connectionType = this.connectionType.clone();

			/* Clone the heart beat connection type */
			if (this.heartbeatConnectionType != null)
				clone.heartbeatConnectionType = this.heartbeatConnectionType.clone();
		}
		catch (CloneNotSupportedException ex)
		{ // Should never happen
		}

		return clone;
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString()
	{
		return this.getName();
	}
}
