/*
 * Created on Sep 6, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.external.wpexport.impl;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExporterInputAttributes;
import com.tandbergtv.watchpoint.studio.external.wpexport.IWatchPointDTOExporter;
import com.tandbergtv.watchpoint.studio.util.FileUtil;

/**
 * Exporter used when creating a WatchPoint plug-in folder and it contents.
 * 
 * @param <T>
 *            The Type of the object to export
 * 
 * @author Vijay Silva
 */
public abstract class WatchPointPluginExporter<T extends IWatchPointDTO> implements
		IWatchPointDTOExporter<T>
{
	// The logger
	private static final Logger logger = Logger.getLogger(WatchPointPluginExporter.class);

	// The name of the plug-in XML file
	protected static final String PLUGIN_FILE_NAME = "plugin.xml";
	
	protected static final String GENERIC_EXCEPTION_MESSAGE = "Ensure that the export folder is accessible.";

	/** Common File Separator to use when building file paths that is supported on all platforms */
	protected static final String FILE_SEPARATOR = "/";

	/**
	 * Creates a Plug-in Folder by using the path specified in the inputs, copies all required
	 * libraries / resources to the new folder, and generates the 'plugin.xml' file.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.external.wpexport.IWatchPointDTOExporter#export(com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO,
	 *      java.util.Map)
	 */
	public Map<String, Object> export(T exportable, Map<String, Object> exportInputs)
			throws ExportFailureException
	{
		try
		{
			// Create / Get the plug-in Folder in which the plug-in contents will be placed.
			File targetFolder = this.getPluginFolder(exportable, exportInputs);

			// Build the plug-in XML document.
			Document document = this.buildPluginDocument(exportable, exportInputs);

			// Add the Plug-in dependencies
			this.addPluginDependencies(exportable, exportInputs, document, targetFolder);

			// Add Runtime environment
			this.handleRuntimeLibraries(exportable, exportInputs, document, targetFolder);

			// Add the Extensions / Extension Points
			this.addPluginPointsAndExtensions(exportable, exportInputs, document, targetFolder);

			// Writes the Plug-in document to the root folder
			this.writePluginDocument(exportable, exportInputs, document, targetFolder);

			return new HashMap<String, Object>();
		}
		catch (ExportFailureException efe)
		{
			throw efe;
		}
		catch (Exception ex)
		{
			String msg = "Failed to export the WatchPoint Plugin.";
			throw new ExportFailureException(msg, ex);
		}
	}

	/**
	 * Gets the Folder in which the plug-in contents will be placed. This folder will be the root
	 * folder for the plug-in. The folder must exist at the end of this operation (create folder if
	 * it doesn't exist). If a folder with the same name as the rootFolderName already exists, the
	 * rootFolderName suffixed with '_<current time stamp>'.
	 * 
	 * @param exportable
	 *            The object to export
	 * @param inputs
	 *            The export process inputs
	 * 
	 * @return The File object for the Plug-in root folder
	 * 
	 * @throws ExportFailureException
	 *             Failure to create / get the Plug-in root folder
	 */
	protected File getPluginFolder(T exportable, Map<String, Object> inputs)
			throws ExportFailureException
	{
		String folderPath = (String) inputs.get(ExporterInputAttributes.EXPORT_FOLDER_PATH);
		if (folderPath == null)
		{
			String msg = "No Output Folder provided for export of WatchPoint Plugin, property: "
					+ ExporterInputAttributes.EXPORT_FOLDER_PATH
					+ " must be assigned with the value of an existing folder path"
					+ "in the export inputs when exporting a WatchPoint plugin.";
			throw new ExportFailureException(msg);
		}

		File pluginFolder = new File(folderPath);
		folderPath = pluginFolder.getAbsolutePath();
		if (!pluginFolder.exists())
		{ // Create the folder if it doesn't exist
			pluginFolder.mkdirs();
		}

		/* Check if the container folder is actually a folder */
		if (!pluginFolder.isDirectory())
		{
			String msg = "The specified output folder for export of WatchPoint Plugin"
					+ " is not a folder: " + folderPath;
			throw new ExportFailureException(msg);
		}

		return pluginFolder;
	}

	/**
	 * Create the XML Document that will contain the contents of the 'plugin.xml' file. The basic
	 * plug-in document created contains the 'DocType' and root 'plugin' element.
	 * 
	 * @param exportable
	 *            The object to export
	 * @param inputs
	 *            The Map of inputs to use when exporting
	 * 
	 * @return The basic plug-in document created
	 * 
	 * @throws ExportFailureException
	 *             Failure to build the Plug-in XML Document
	 */
	protected Document buildPluginDocument(T exportable, Map<String, Object> inputs)
			throws ExportFailureException
	{
		String pluginId = this.getPluginId(exportable, inputs);
		String version = this.getPluginVersion(exportable, inputs);

		try
		{
			return JPFExportUtil.createPluginDocument(pluginId, version);
		}
		catch (ParserConfigurationException ex)
		{
			String msg = "Export failed, unable to build the plugin XML document.";
			throw new ExportFailureException(msg, ex);
		}
	}

	/**
	 * Add the required Plug-in dependencies for this current plug-in in the input plug-in document.
	 * Currently, just the WatchPoint Plug-in is required as a dependency requirement.
	 * 
	 * @param exportable
	 *            The object to export
	 * @param inputs
	 *            The export process inputs
	 * @param pluginDocument
	 *            The Plug-in document
	 * @param pluginRootFolder
	 *            The root folder for the Plug-in
	 */
	protected void addPluginDependencies(T exportable, Map<String, Object> inputs,
			Document pluginDocument, File pluginRootFolder)
	{
		// Add the dependency to the WatchPoint Plug-in
		JPFExportUtil.addPluginDependency(pluginDocument, WatchPointPluginConstants.PLUGIN_ID);
	}

	/**
	 * Copies all the required runtime libraries to the plug-in root folder and adds all the
	 * libraries to the runtime section of the plug-in document.
	 * 
	 * @param exportable
	 *            The object to export
	 * @param inputs
	 *            The export process inputs
	 * @param pluginDocument
	 *            The Plug-in XML Document
	 * @param pluginRootFolder
	 *            The Root Folder for the plug-in
	 * 
	 * @throws ExportFailureException
	 *             Exception thrown in case of failure to copy the runtime libraries, or write the
	 *             libraries to the plug-in document.
	 */
	@SuppressWarnings("unchecked")
	protected void handleRuntimeLibraries(T exportable, Map<String, Object> inputs,
			Document pluginDocument, File pluginRootFolder) throws ExportFailureException
	{
		List<String> codeLibraries = new ArrayList<String>();
		List<String> resourceLibraries = new ArrayList<String>();

		/* Copy all the archived libraries */
		String folderName = WatchPointPluginConstants.PLUGIN_LIB_FOLDER_NAME;
		Object value = inputs.get(ExporterInputAttributes.PLUGIN_JAR_FILE_PATHS);
		List<String> filePaths = (List<String>) value;
		this.copyRuntimeFiles(pluginRootFolder, folderName, filePaths, codeLibraries, true);

		/* Copy all the class files */
		folderName = WatchPointPluginConstants.PLUGIN_CLASS_FOLDER_NAME;
		value = inputs.get(ExporterInputAttributes.PLUGIN_CLASSES_FILE_PATHS);
		filePaths = (List<String>) value;
		this.copyRuntimeFiles(pluginRootFolder, folderName, filePaths, codeLibraries, false);

		/* Copy all the resource files */
		folderName = WatchPointPluginConstants.PLUGIN_RESOURCE_FOLDER_NAME;
		value = inputs.get(ExporterInputAttributes.PLUGIN_RESOURCE_FILE_PATHS);
		filePaths = (List<String>) value;
		this.copyRuntimeFiles(pluginRootFolder, folderName, filePaths, resourceLibraries, false);

		JPFExportUtil.addRuntimeLibraries(pluginDocument, codeLibraries, resourceLibraries);
	}

	/**
	 * Write the plug-in document to file 'plugin.xml' in the root folder for the plug-in
	 * 
	 * @param exportable
	 *            The object to export
	 * @param inputs
	 *            The export process inputs
	 * @param pluginDocument
	 *            The plug-in XML Document
	 * @param pluginRootFolder
	 *            The root folder for the plug-in
	 * 
	 * @throws ExportFailureException
	 *             Exception writing the plug-in document
	 */
	protected void writePluginDocument(T exportable, Map<String, Object> inputs,
			Document pluginDocument, File pluginRootFolder) throws ExportFailureException
	{
		String filePath = null;
		Writer writer = null;

		try
		{
			File targetFile = new File(pluginRootFolder, PLUGIN_FILE_NAME);
			filePath = targetFile.getAbsolutePath();

			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer();

			// Set the DocType and indentation
			transformer.setOutputProperty(OutputKeys.DOCTYPE_PUBLIC, JPFConstants.DOCTYPE_PUBLIC_ID);
			transformer.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM, JPFConstants.DOCTYPE_SYSTEM_ID);
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");

			writer = new OutputStreamWriter(new FileOutputStream(targetFile), "UTF-8");
			transformer.transform(new DOMSource(pluginDocument), new StreamResult(writer));
		}
		catch (Exception ex)
		{
			String msg = "Failed to write the Plugin document to the file: " + filePath + ". " + GENERIC_EXCEPTION_MESSAGE;
			throw new ExportFailureException(msg, ex);
		}
		finally
		{
			if (writer != null)
			{
				try
				{
					writer.close();
				}
				catch (IOException ex)
				{
					String msg = "Failed to close file stream after writing WatchPoint Plugin file: "
							+ filePath;
					logger.warn(msg, ex);
				}
			}
		}
	}

	/**
	 * Get the Folder to use as the Plug-in Library Folder
	 * 
	 * @param pluginRootFolder
	 *            The Plug-in Root folder
	 * @return A File pointing to the Plug-in Library Folder
	 */
	protected final File getPluginLibFolder(File pluginRootFolder)
	{
		String folderName = WatchPointPluginConstants.PLUGIN_LIB_FOLDER_NAME;
		return this.getLibraryFolder(pluginRootFolder, folderName);
	}

	/**
	 * Get the Folder to use as the Plug-in Classes Folder
	 * 
	 * @param pluginRootFolder
	 *            The Plug-in Root folder
	 * @return A File pointing to the Plug-in Classes Folder
	 */
	protected final File getPluginClassesFolder(File pluginRootFolder)
	{
		String folderName = WatchPointPluginConstants.PLUGIN_CLASS_FOLDER_NAME;
		return this.getLibraryFolder(pluginRootFolder, folderName);
	}

	/**
	 * Get the Folder to use as the Plug-in Resources Folder
	 * 
	 * @param pluginRootFolder
	 *            The Plug-in Root folder
	 * @return A File pointing to the Plug-in Resources Folder
	 */
	protected final File getPluginResourcesFolder(File pluginRootFolder)
	{
		String folderName = WatchPointPluginConstants.PLUGIN_RESOURCE_FOLDER_NAME;
		return this.getLibraryFolder(pluginRootFolder, folderName);
	}

	// ========================================================================
	// ===================== ABSTRACT METHODS
	// ========================================================================

	/**
	 * Get the Plug-In Id of the new plug-in being created.
	 * 
	 * @param exportable
	 *            The object to export
	 * @param inputs
	 *            The export process inputs
	 * 
	 * @return The Plug-in Id
	 */
	protected abstract String getPluginId(T exportable, Map<String, Object> inputs);

	/**
	 * Get the Plug-In version of the new plug-in being created.
	 * 
	 * @param exportable
	 *            The object to export
	 * @param inputs
	 *            The export process inputs
	 * 
	 * @return The Plug-in version
	 */
	protected abstract String getPluginVersion(T exportable, Map<String, Object> inputs);

	/**
	 * Adds all required extension and extension points in the plug-in document for the object being
	 * exported.
	 * 
	 * @param exportable
	 *            The object to export
	 * @param inputs
	 *            The export process inputs
	 * @param pluginDocument
	 *            The Plug-in XML Document
	 * @param pluginRootFolder
	 *            The Root Folder for the plug-in
	 * 
	 * @throws ExportFailureException
	 *             Exception thrown in case the extensions / extension points could not be created
	 *             and added.
	 */
	protected abstract void addPluginPointsAndExtensions(T exportable, Map<String, Object> inputs,
			Document pluginDocument, File pluginRootFolder) throws ExportFailureException;

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/*
	 * Get the Library Folder, given the Root Folder, and the relative library folder path
	 */
	private File getLibraryFolder(File pluginRootFolder, String libraryFolderName)
	{
		/* Get the file for the library folder */
		String folderName = (libraryFolderName == null) ? "" : libraryFolderName.trim();
		File libraryFolder = (folderName.length() > 0) ? new File(pluginRootFolder, folderName)
				: pluginRootFolder;

		if (libraryFolder.exists())
			FileUtil.deleteFolderContents(libraryFolder, true);
		
		/* Create the library folder, if required */
		if (!libraryFolder.exists())
			libraryFolder.mkdirs();

		return libraryFolder;
	}

	/*
	 * Copy all library paths specified to the libraryFolderName (relative to the pluginRootFolder)
	 * using flag 'onlyFiles' to indicate that only files are expected to be copied, or if both
	 * files and folders are expected to be copied.
	 */
	private void copyRuntimeFiles(File pluginRootFolder, String libraryFolderName,
			List<String> libraryPaths, List<String> currentLibraries, boolean isJarFiles)
			throws ExportFailureException
	{
		if (libraryPaths == null)
		{ // No library files to copy
			libraryPaths = new ArrayList<String>();
		}

		String absolutePluginPath = pluginRootFolder.getAbsolutePath();
		int absolutePathSize = absolutePluginPath.length();
		if (!absolutePluginPath.endsWith(File.separator))
			absolutePathSize++;

		/* Build and create the library folder, if required */
		File libraryFolder = this.getLibraryFolder(pluginRootFolder, libraryFolderName);

		/* Copy all the library files / folders, and keep track the relative copied locations */
		for (String libraryFilePath : libraryPaths)
		{
			File libraryFile = new File(libraryFilePath);
			File target = new File(libraryFolder, libraryFile.getName());

			if (isJarFiles && !libraryFile.isFile())
			{
				String msg = "Failure when attempting to copy file: "
						+ libraryFile.getAbsolutePath() + ", source is not a file.";
				throw new ExportFailureException(msg);
			}

			if (!FileUtil.overwrite(libraryFile, target, true))
			{
				String msg = "Failed to copy file/folder: " + libraryFile.getAbsolutePath()
						+ " to location: " + target.getAbsolutePath()
						+ " required for export of WatchPoint Plugin. " + GENERIC_EXCEPTION_MESSAGE;
				throw new ExportFailureException(msg);
			}

			if (isJarFiles)
			{
				// Get the relative file path from the plug-in root folder
				String absoluteLibraryPath = target.getAbsolutePath();
				String relativePath = absoluteLibraryPath.substring(absolutePathSize);
				this.addJPFCompliantLibraryPath(relativePath, currentLibraries);
			}
		}

		if (!isJarFiles)
		{
			String absoluteLibraryPath = libraryFolder.getAbsolutePath();
			String relativePath = (absolutePathSize >= absoluteLibraryPath.length())
					? File.separator : absoluteLibraryPath.substring(absolutePathSize);
			if (!relativePath.endsWith(File.separator))
				relativePath += File.separator;
			this.addJPFCompliantLibraryPath(relativePath, currentLibraries);
		}
	}

	/*
	 * Add the library path to the list of libraries maintained (unless the path is already
	 * present). Ensures that the path will be accepted by JPF and will work both on Unix and
	 * Windows environments, by replacing all '\' with '/'.
	 */
	private void addJPFCompliantLibraryPath(String path, List<String> libraries) 
	{
		if (path == null)
			return;

		path = path.replaceAll("\\\\", FILE_SEPARATOR);
		if (!libraries.contains(path))
			libraries.add(path);
	}
}
