package com.tandbergtv.watchpoint.studio.service.impl;

import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.GPD_EXTENSION;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.PLUGIN_NODE_FOLDER;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.PLUGIN_RESOURCE_FOLDER_NAME;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.XML_EXTENSION;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.RESOURCE_TYPE_PLUGIN_FILE_NAME;
import static java.io.File.separator;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.w3c.dom.Document;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.external.fs.dao.ResourceTypeDAO;
import com.tandbergtv.watchpoint.studio.external.jpf.JPFContext;
import com.tandbergtv.watchpoint.studio.service.ServiceErrorCode;
import com.tandbergtv.watchpoint.studio.service.ServiceException;
import com.tandbergtv.watchpoint.studio.util.ResourceTypePluginFileGenerator;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;

/**
 * Class responsible to interact with the file system regarding the 
 * resource type operations.
 * @author xpatrpe
 *
 */
public class ResourceTypeFileSystemManager {
	private NodeDefinitionFileSystemWriter nodeDefinitionWriter;
	
	public ResourceTypeFileSystemManager() {
		this.nodeDefinitionWriter = new NodeDefinitionFileSystemWriter();
	}
	
	/**
	 * Update the resource type plugin.xml file in file system.<br>
	 * Basically a new plugin.xml file is generated and it 
	 * overwrites the existing plugin.xml file.<br>
	 * This method doesn't handle with node definitions. Only information regarding the 
	 * plugin.xml file.
	 * @param resourceType The resource type to be updated
	 */
	public void updateResourceType(ResourceType resourceType) throws Exception{
		Document pluginDocument = ResourceTypePluginFileGenerator.generatePluginDocument(resourceType);
		ByteArrayOutputStream plugin = ResourceTypePluginFileGenerator.createWritablePluginDocument(pluginDocument, new File(""));
		byte[] pluginContentArray = plugin.toByteArray();
		IFile pluginFile = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(resourceType.getPath()));
		// Check if the file is sync with the filesystem.
		// If not, the setContents operation below will throw an exception
		if (!pluginFile.isSynchronized(IResource.DEPTH_ONE)) {
			pluginFile.refreshLocal(IResource.DEPTH_ONE, null);
		}
				
		pluginFile.setContents(new ByteArrayInputStream(pluginContentArray), IFile.KEEP_HISTORY, null);
	}
	
	/**
	 * Create a node definition if file system for a given resource type.<br>
	 * This method writes to the file system the content of the xml and gpd attributes from the 
	 * node definition object.
	 * @param nodeDTO The node definition to be created.
	 * @param resourceType The corresponding resource type for this node.
	 * @throws Exception 
	 */
	public void createNodeDefinition(NodeDefinitionDTO nodeDTO) throws Exception{
		Collection<NodeDefinitionDTO> nodes = new ArrayList<NodeDefinitionDTO>();
		nodes.add(nodeDTO);
		IProject proj = ResourcesPlugin.getWorkspace().getRoot().getProject(nodeDTO.getResourceType().determineNameForProject());
		
		nodeDefinitionWriter.writeNodeDefinitionsInProject(nodes, proj);
	}
	
	/**
	 * Deletes the node definition from the file system.<br>
	 * This method deletes the .gpd and the .xml files for the given node definition.
	 * @param nodeDTO The node definition to be deleted.
	 * @param resourceType The corresponding resource type for this node.
	 * @throws CoreException 
	 */
	public void deleteNodeDefinition(NodeDefinitionDTO nodeDTO) throws CoreException{
		IProject proj = ResourcesPlugin.getWorkspace().getRoot().getProject(nodeDTO.getResourceType().determineNameForProject());
		
		String nodePath = separator + "resources" + separator + "node"+ separator;
		proj.getFile(nodePath + nodeDTO.getName() + ".xml").delete(true,true, null);
		proj.getFile(nodePath + nodeDTO.getName() + ".gpd").delete(true,true, null);
	}
	
	public ResourceType getResourceType(String projectName){
		IProject proj = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		if(proj.exists()){
			IFile file = proj.getFile(RESOURCE_TYPE_PLUGIN_FILE_NAME);
			JPFContext context = new JPFContext(file.getParent().getLocation().toOSString());
			context.initialize();
			ResourceTypeDAO dao = new ResourceTypeDAO(context);
			List<ResourceType> resourceTypes = dao.findAll();
			
			ResourceType parsed = resourceTypes.get(0);
			parsed.setPath(file.getFullPath().toPortableString());
			parsed.setProjectName(proj.getName());
			return parsed;
		}
		return null;
	}
	
	public void deleteResourceType(ResourceType resourceType){
		final IProject proj = ResourcesPlugin.getWorkspace().getRoot().getProject(resourceType.determineNameForProject());
		Job job = new Job("Deleting resource type project") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				try {
					monitor.beginTask("Delete project", 2);
					proj.delete(true, true, monitor);
					monitor.worked(1);
					monitor.done();
				} catch (CoreException e) {
					throw new ServiceException(ServiceErrorCode.GENERAL_ERROR, "Error deleting resource type project", e);
				} 
				return Status.OK_STATUS;
			}
		};
		job.setUser(true);
		job.schedule();
	}
	
	public void updateNodeDefiniton(NodeDefinitionDTO nodeDTO) throws CoreException, Exception{
		IProject proj = ResourcesPlugin.getWorkspace().getRoot().getProject(nodeDTO.getResourceType().determineNameForProject());
		
		String nodeFolderPath = separator + PLUGIN_RESOURCE_FOLDER_NAME + separator + PLUGIN_NODE_FOLDER;
		String xmlFilePath = nodeFolderPath + separator + nodeDTO.getName() + XML_EXTENSION;
		String gpdFilePath = nodeFolderPath + separator + nodeDTO.getName() + GPD_EXTENSION;
		
		if(proj.getFile(xmlFilePath).exists()){
			overwriteNodeDefinitionFile(nodeDTO.getXml(), xmlFilePath, proj);
		}
		if(proj.getFile(gpdFilePath).exists()){
			overwriteNodeDefinitionFile(nodeDTO.getGpd(), gpdFilePath, proj);
		}
	}
	
	private void overwriteNodeDefinitionFile(String xml, String projectPath, IProject proj) throws Exception{
		Document doc = XMLDocumentUtility.loadXml(xml);
		ByteArrayOutputStream baos = XMLDocumentUtility.createByteArray(doc);
		byte[] buildContentArray = baos.toByteArray();
		IFile file = proj.getFile(projectPath);
		file.setContents(new ByteArrayInputStream(buildContentArray), true, false, null);
	}
	
}
