/*
 * Created on Sep 24, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart;

import static com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType.MessageNode;
import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.TIMEOUT_WRONG_FORMAT_MESSAGE;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.forms.AbstractFormPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.common.xml.XmlAdapter;

import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.INodeDefinitionService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.ResourceTypeEditorInput;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.eventhandler.FormPartDirtyModifyListener;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.eventhandler.TextTrimFocusListener;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;
import com.tandbergtv.watchpoint.studio.util.TimeoutValidation;

/**
 * A SectionPart that is used to display ResourceType information.
 * 
 * @author Vijay Silva
 */
class MessageDetailsSectionPart extends ResourceTypeSectionPart
{
	private static final String INCOMING_TYPE = "Incoming";

	private static final String OUTGOING_TYPE = "Outgoing";

	Label nameLabel, uidLabel, typeLabel, timeoutLabel, descriptionLabel;

	Text nameText, uidText, timeoutText, descriptionText;
	
	ResourceTypeEditorInput resourceTypeInput;

	CCombo typeCombo;

	private AbstractFormPart masterPart;

	// ========================================================================
	// ===================== CONSTRUCTORS
	// ========================================================================

	/**
	 * @param parentPart
	 *            The Part that contains this composite
	 * @param parent
	 *            The Parent Composite
	 * @param toolkit
	 *            The toolkit to create the widgets
	 * @param style
	 *            The Style bits to create the Section
	 */
	public MessageDetailsSectionPart(IEditorPart parentPart, Composite parent, FormToolkit toolkit,
			int style)
	{
		super(parentPart, parent, toolkit, style);
	}

	/**
	 * Set the Master Part which needs refreshing when the changes to this part are committed.
	 * 
	 * @param masterPart
	 *            The Master Section Part
	 */
	public void setMessageMasterPart(AbstractFormPart masterPart)
	{
		this.masterPart = masterPart;
	}

	// ========================================================================
	// ===================== OVERRIDDEN METHODS
	// ========================================================================

	/**
	 * @see org.eclipse.ui.forms.AbstractFormPart#setFormInput(java.lang.Object)
	 */
	@Override
	public boolean setFormInput(Object input)
	{
		if (input instanceof MessageInput)
		{
			this.input = (MessageInput) input;
			this.markStale();

			return true;
		}

		return false;
	}

	/**
	 * @see org.eclipse.ui.forms.AbstractFormPart#commit(boolean)
	 */
	@Override
	public void commit(boolean onSave)
	{
		super.commit(onSave);

		Message message = this.getMessage();
		
		if (message == null)
			return;
		
		message.setName(this.nameText.getText());
		message.setProtocol(null);
		message.setDescription(this.descriptionText.getText());
		
		NodeDefinitionDTO dto = getNodeDefinition(message);
		
		if (dto == null)
			return;
		
		NodeDefinition node = getNodeDefinition(dto);
		
		SemanticElementFactory factory = ((SemanticElement) node).getFactory();
		XmlAdapter adapter = Utility.getAdapterFromString(dto.getXml(), factory);
		
		node.setId(dto.getId());
		adapter.initialize(node);
		
		
		// Check the time out expression
		if( !TimeoutValidation.isTimeoutExpressionValid(this.timeoutText.getText()) ){
			MessageDialog.openError(Display.getCurrent().getActiveShell(), "Error",
					TIMEOUT_WRONG_FORMAT_MESSAGE);
			return ;
		}
		
		node.setDueDate(this.timeoutText.getText());
		
		String xml = Utility.getXMLFromDocument(adapter.getNode().getOwnerDocument());
		
		dto.setXml(xml);
		
		//Input added to make possible to get the resource type object and change it
		ResourceType rt = resourceTypeInput.getResourceType();
		
		Set<NodeDefinitionDTO> tempCopy = new HashSet<NodeDefinitionDTO>(rt.getNodes());
		
		// Update the Resource Type object managed by the editor
		// Set the modified node definition
		for (NodeDefinitionDTO nodeDTO : tempCopy) {
			if(nodeDTO.getName().equals(dto.getName())){
				rt.getNodes().remove(nodeDTO);
			}
		}
		rt.getNodes().add(dto);
		
		INodeDefinitionService nodeDefinitionService = ServiceFactory.createFactory().createNodeDefinitionService();
		nodeDefinitionService.updateNodeDefinitionInFileSystem(dto);
		
		// Refresh the master section
		if (masterPart != null)
			masterPart.markStale();
	}
	
	private NodeDefinition getNodeDefinition(NodeDefinitionDTO dto)
	{
		try
		{
			return SemanticElementUtil.createNodeDefinition(dto.getXml());
		}
		catch (Exception e)
		{
			throw new RuntimeException();
		}
	}
	
	private NodeDefinitionDTO getNodeDefinition(Message message)
	{
		for (NodeDefinitionDTO dto : message.getResourceType().getNodes())
		{
			if (dto.getType() != MessageNode)
				continue;
			
			try
			{
				NodeDefinition node = SemanticElementUtil.createNodeDefinition(dto.getXml());
				
				if (message.getUid().equals(node.getUid()))
					return dto;
			}
			catch (Exception e)
			{
				continue;
			}
		}
		
		return null;
	}

	// ========================================================================
	// ===================== USER CONTROL INITIALIZATION
	// ========================================================================

	/**
	 * Create the Controls displayed in the composite
	 * 
	 * @param managedForm
	 *            The ManagedForm that will contain this Form Part
	 */
	protected void createSectionClientControls(IManagedForm managedForm)
	{
		FormToolkit toolkit = managedForm.getToolkit();

		/* The Name Label and Text */
		this.nameLabel = toolkit.createLabel(this.sectionClient, "Name: ");
		this.nameText = toolkit.createText(this.sectionClient, "");
		this.nameText.setTextLimit(50);
		this.nameText.addFocusListener(new TextTrimFocusListener());
		this.nameText.addModifyListener(new FormPartDirtyModifyListener(this));

		/* The UID Label and Text */
		this.uidLabel = toolkit.createLabel(this.sectionClient, "UID: ");
		this.uidText = toolkit.createText(this.sectionClient, "", SWT.READ_ONLY);
		this.uidText.setTextLimit(10);
		this.uidText.addFocusListener(new TextTrimFocusListener());
		this.uidText.addModifyListener(new FormPartDirtyModifyListener(this));

		/* The Type Label and Combo */
		this.typeLabel = toolkit.createLabel(this.sectionClient, "Type: ");
		int style = SWT.READ_ONLY | toolkit.getBorderStyle();
		this.typeCombo = new CCombo(this.sectionClient, style);
		toolkit.adapt(this.typeCombo, true, true);
		this.typeCombo.setEnabled(false);
		this.typeCombo.add(INCOMING_TYPE, 0);
		this.typeCombo.add(OUTGOING_TYPE, 1);
		
		/* The Timeout Label and Text */
		this.timeoutLabel = toolkit.createLabel(this.sectionClient, "Time Out: ");
		this.timeoutText = toolkit.createText(this.sectionClient, "", SWT.NONE);
		this.timeoutText.setTextLimit(11);
		this.timeoutText.addFocusListener(new TextTrimFocusListener());
		this.timeoutText.addModifyListener(new FormPartDirtyModifyListener(this));

		/* The Description Label and Text */
		this.descriptionLabel = toolkit.createLabel(this.sectionClient, "Description: ");
		this.descriptionText = toolkit.createText(this.sectionClient, "", SWT.MULTI | SWT.V_SCROLL
				| SWT.H_SCROLL);
		this.descriptionText.setTextLimit(255);
		this.descriptionText.addModifyListener(new FormPartDirtyModifyListener(this));
	}

	/**
	 * Initialize the Layout for the Section Client and its child widgets
	 * 
	 * @param managedForm
	 *            The Managed Form that contains this form Part
	 */
	protected void initializeSectionClientLayout(IManagedForm managedForm)
	{
		/* Set the Layout for the Section Client in the Form */
		this.sectionClient.setLayout(this.createGridLayout(2, false));

		/* Set the default Layout data for the widgets */
		this.nameLabel.setLayoutData(this.createLabelGridData());
		this.uidLabel.setLayoutData(this.createLabelGridData());
		this.typeLabel.setLayoutData(this.createLabelGridData());
		this.timeoutLabel.setLayoutData(this.createLabelGridData());
		this.descriptionLabel.setLayoutData(this.createLabelGridData());

		this.uidText.setLayoutData(this.createGridData());
		this.timeoutText.setLayoutData(this.createGridData());
		this.typeCombo.setLayoutData(this.createGridData());

		/* Set the Layout data for the Name, Protocol and Description Text */
		GridData layoutData = this.createGridData();
		layoutData.widthHint = 150;
		this.nameText.setLayoutData(layoutData);

		layoutData = new GridData(SWT.FILL, SWT.FILL, true, true);
		layoutData.heightHint = 250;
		this.descriptionText.setLayoutData(layoutData);
	}

	// ========================================================================
	// ===================== DISPLAYING THE INPUT DATA
	// ========================================================================

	/*
	 * Display the Input data in the Section Widgets
	 */
	protected void populateSectionData()
	{
		MessageInput input = (MessageInput) this.input;
		Message message =  input != null ? input.getMessage() : null;

		if (message != null)
		{
			this.nameText.setText(this.formatValue(message.getName(), true));
			this.nameText.setEnabled(false);
			this.uidText.setText(this.formatValue(message.getUid(), true));			
			this.typeCombo.setText(message.isIncoming() ? INCOMING_TYPE : OUTGOING_TYPE);
			this.descriptionText.setText(this.formatValue(message.getDescription(), false));
			
			if (input.getTimeout() != null)
			{
				this.timeoutText.setText(input.getTimeout());
			}
			else
				this.timeoutText.setText("");
		}
		else
		{
			this.nameText.setText("");
			this.uidText.setText("");
			this.timeoutText.setText("");
			this.typeCombo.setText("");
			this.descriptionText.setText("");
		}
	}

	/*
	 * Set the permissions of the section widgets based on the input
	 */
	protected void setWidgetPermissions()
	{
		boolean editable = this.isInputEditable();
		this.nameText.setEditable(editable);
		this.descriptionText.setEditable(editable);
	}
	
	public void setResourceTypeInput(ResourceTypeEditorInput input){
		this.resourceTypeInput = input;
	}
}
