package com.tandbergtv.watchpoint.studio.ui.sync.compare;

import static com.tandbergtv.watchpoint.studio.ui.sync.DiffKind.ADD;
import static com.tandbergtv.watchpoint.studio.ui.sync.DiffKind.CHANGE;
import static com.tandbergtv.watchpoint.studio.ui.sync.DiffKind.REMOVE;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;

import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITree;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITreeNode;
import com.tandbergtv.watchpoint.studio.ui.model.tree.provider.MutableTreeNode;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.SynchronizationException;
import com.tandbergtv.watchpoint.studio.ui.sync.impl.Diff;

/**
 * 		Comparer for the Rollback Node Definition.
 * 
 * @author <a href="francisco.neto@venturus.org.br">vntfrne Francisco Bento da Silva Neto</a>
 *
 */
public class RollbackNodeDefinitionComparer extends NodeDefinitionComparer {

	/**
	 * {@inheritDoc}
	 */
	public ITree<IDiff> compare(NodeDefinition local, NodeDefinition remote, IProgressMonitor monitor) {
		this.localElement = local;
		this.remoteElement = remote;
		
		checkRename();
		
		ITree<IDiff> tree = doCompare(monitor);
		
		return tree;
	}
	
	protected ITree<IDiff> doCompare(IProgressMonitor monitor) {
		ITree<IDiff> tree = createDiffTree();
		ITreeNode<IDiff> root = tree.getRoot();
		
		try {
			if (monitor != null) {
				monitor.beginTask("Calculating changes to variables", 3);
			}
			
			for (ITreeNode<IDiff> treenode : getAddedTreeNodes())
				root.addChild(treenode);
			
			incrementMonitor(monitor);
			
			for (ITreeNode<IDiff> treenode : getRemovedTreeNodes())
				root.addChild(treenode);
			
			incrementMonitor(monitor);
			
			for (ITreeNode<IDiff> treenode : getChangedTreeNodes())
				root.addChild(treenode);
			
			incrementMonitor(monitor);
			
			if (!root.hasChildren())
				return createEmptyTree(local);
		} finally {
			if (monitor != null) {
				monitor.done();
			}
		}
		
		return tree;
	}
	
	protected void checkRename() {
		boolean match = localElement.getRollbackUid().equals(remoteElement.getUid());
		if (!match) {
			throw new SynchronizationException("Node " + local.getName() + " - uid mismatch, current [" + 
				localElement.getUid() + "], new [" + remoteElement.getUid() + "]");
		}
	}
	
	/**
	 * Gets the tree nodes containing diffs that represent variables that have been added to the node,
	 * including composite keys.
	 * 
	 * @return
	 */
	protected List<ITreeNode<IDiff>> getAddedTreeNodes() {
		List<ITreeNode<IDiff>> treenodes = new ArrayList<ITreeNode<IDiff>>();

		for (WPVariable remoteVariable : remoteElement.getVariables()) {
			boolean hasVariable = localElement.hasRollbackVariable(remoteVariable.getMappedName());	
			if (!hasVariable) {
				treenodes.add(new MutableTreeNode<IDiff>(new Diff(remoteVariable, ADD)));
			}
		}
		return treenodes;
	}
	
	/**
	 * Gets the tree nodes containing diffs that represent variables that have been removed from the node,
	 * including composite keys.
	 * 
	 * @return
	 */
	protected List<ITreeNode<IDiff>> getRemovedTreeNodes() {
		List<ITreeNode<IDiff>> treenodes = new ArrayList<ITreeNode<IDiff>>();
		
		List<WPVariable> variables = localElement.getRollbackVariables();
		for (WPVariable localVariable : variables) {
			if (!remoteElement.hasVariable(localVariable.getMappedName()))
				treenodes.add(new MutableTreeNode<IDiff>(new Diff(localVariable, REMOVE)));
		}
	
		return treenodes;
	}
	
	/**
	 * @return
	 */
	protected List<ITreeNode<IDiff>> getChangedTreeNodes() {
		List<ITreeNode<IDiff>> treenodes = new ArrayList<ITreeNode<IDiff>>();
		
		List<WPVariable> variables = localElement.getRollbackVariables();
		for (WPVariable localVariable : variables) {
			if (localVariable.isCompositeKey())
				continue;
			
			if (!remoteElement.hasVariable(localVariable.getMappedName()))
				continue;
			
			WPVariable remoteVariable = remoteElement.getVariable(localVariable.getMappedName());
			
			if (hasAccessChanged(localVariable, remoteVariable))
				treenodes.add(new MutableTreeNode<IDiff>(new Diff(remoteVariable, "access", CHANGE)));
			
			if (hasTypeChanged(localVariable, remoteVariable))
				treenodes.add(new MutableTreeNode<IDiff>(new Diff(remoteVariable, "type", CHANGE)));
		}
		
		return treenodes;
	}
}
