/**
 * TemplateDiffCommand.java
 * Created Apr 21, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.sync.template;

import static com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants.TEMPLATE_SEID;
import static com.tandbergtv.watchpoint.studio.ui.util.Utility.TEMPLATE_EDITOR_ID;

import java.util.Iterator;

import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.common.xml.XmlAdapter;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.w3c.dom.Node;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.TemplateMessageDTO;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.command.CompositeCommand;
import com.tandbergtv.watchpoint.studio.ui.command.ICommand;
import com.tandbergtv.watchpoint.studio.ui.model.IResourceGroupElement;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.sync.util.ISynchronizationContext;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * A command that encapsulates bulk operations to a template. It contains other commands that modify
 * the template in some fashion. Commands that use the {@link ISynchronizationContext} to perform
 * updates must *not* be added.
 * 
 * @author Sahil Verma
 */
public class TemplateDiffCommand extends CompositeCommand {

	/**
	 * The model undergoing change 
	 */
	protected WorkflowTemplateDTO model;

	/**
	 * The UI model
	 */
	protected WorkflowTemplate container;

	/**
	 * Adapts the UI model to XML
	 */
	protected XmlAdapter adapter;

	protected ISynchronizationContext context;

	/**
	 * @param name
	 * @param description
	 */
	public TemplateDiffCommand(String name, String description, ISynchronizationContext context) {
		super(name, description);
		this.context = context;
	}

	/**
	 * @return the model
	 */
	public WorkflowTemplateDTO getModel() {
		return model;
	}

	/**
	 * @param model the model to set
	 */
	public void setModel(WorkflowTemplateDTO model) {
		this.model = model;
		this.container = getSemanticElement();

		this.adapter = getAdapter();
		this.adapter.initialize(this.container);
	}

	/**
	 * @return the container
	 */
	public WorkflowTemplate getNodeElementContainer() {
		return container;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addCommand(ICommand command) {
		/*
		 * Ensure that commands related to superstate nodes execute before we start adding, removing
		 * or changing superstate variables
		 */
		if (command instanceof AbstractVariableCommand)
			super.addCommand(command);
		else
			getCommands().add(0, command);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void execute() {
		super.execute();

		updateRelationships();

		String xml = getXml(adapter.getNode());

		model.setXml(xml);

		context.updateTemplate(model);
	}

	protected void updateRelationships() {
		removeUnusedResourceGroups();
	}

	protected boolean isUsedInTemplate(NodeDefinitionDTO child) {
		return !container.findNodesRecursively(child.getId()).isEmpty();
	}

	/**
	 * Remove relationship with resource groups that are not used anymore
	 */
	protected void removeUnusedResourceGroups() {
		Iterator<TemplateMessageDTO> i = model.getMessages().iterator();
		
		while (i.hasNext()) {
			TemplateMessageDTO message = i.next();
			if (message.getResourceGroupName() != null && !message.getResourceGroupName().isEmpty()) {
				continue;
			}
			boolean exists = false;
			
			for (NodeElement e : container.getNodeElements()) {
				if (e instanceof IResourceGroupElement) {
					IResourceGroupElement rge = IResourceGroupElement.class.cast(e);
					com.tandbergtv.watchpoint.studio.ui.model.ResourceGroup g = rge.getResourceGroup();
					
					exists = exists || (g != null && message.getResourceGroupName().equals(g.getName()));
				}
			}
			
			if (!exists)
				i.remove();
		}
	}

	protected String getXml(Node node) {
		return Utility.getXMLFromDocument(node.getOwnerDocument());
	}

	protected XmlAdapter getAdapter() {
		SemanticElementFactory factory = new SemanticElementFactory(TEMPLATE_EDITOR_ID);
		try {
			XmlAdapter adapter = Utility.getAdapterFromString(model.getXml(), factory);
			return adapter;
		} catch (RuntimeException e) {
			throw new UIException("Error while parsing template from database: " + model.getName() + " : " + e.getMessage(), e);
		}
	}

	protected WorkflowTemplate getSemanticElement() {
		return (WorkflowTemplate) new SemanticElementFactory(TEMPLATE_EDITOR_ID).createById(TEMPLATE_SEID);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();

		for (ICommand command : getCommands())
			sb.append(command).append(System.getProperty("line.separator"));

		return sb.toString();
	}
}
