/*
 * Created on Jul 31, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.util;

import static com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants.NODE_DEFN_SEID;
import static com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants.TEMPLATE_SEID;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.common.xml.SimpleDOMAdapterFactory;
import org.jbpm.gd.common.xml.XmlAdapter;
import org.jbpm.gd.common.xml.XmlAdapterFactory;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.Transition;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * Utility Class for UI Model Loading
 * 
 * @author Vijay Silva
 */
public class SemanticElementUtil
{
	/**
	 * The Output Map Key that contains the adapter when building the Model element from Xml. The
	 * type of the map value for this key is XmlAdapter.
	 * 
	 * @see org.jbpm.gd.common.xml.XmlAdapter
	 */
	public static final String ADAPTER_KEY = "Adapter";

	/**
	 * The Output Map Key containing the adapter factory when building the Model element from Xml.
	 * The type of the map value for this key is XmlAdapterFactory.
	 * 
	 * @see org.jbpm.gd.common.xml.XmlAdapterFactory
	 */
	public static final String ADAPTER_FACTORY_KEY = "AdapterFactory";

	/**
	 * The Output Map Key for the Semantic Element constructed from Xml. The type of the map value
	 * for this key is SemanticElement.
	 * 
	 * @see org.jbpm.gd.common.model.SemanticElement
	 */
	public static final String SEMANTIC_ELEMENT_KEY = "SemanticElement";

	/**
	 * The Transition Path Seperator used when generating transition paths coming in or out of a
	 * Super State.
	 */
	public static final String TRANSITION_PATH_SEPARATOR = "/";

	/**
	 * Creates the Semantic Element represented in the XML String given the Semantic Element Id.
	 * This method is not thread-safe since it uses the ModelManagerImpl which caches the models
	 * created for each specified key. To ensure thread-safety, ensuring lock for each model key
	 * specified.
	 * 
	 * @param xmlString
	 *            The XML representation of the model object
	 * @param semanticElementId
	 *            The Semantic Element Id of the model object
	 * 
	 * @return A Map containing the Semantic Element constructed, the Adapter for the element, and
	 *         the Adapter Factory used to construct the adapter.
	 * 
	 * @throws IOException
	 *             Failed to build the Model Element from the XML String
	 * @throws UnsupportedEncodingException
	 *             Failed to build the Model Element from the XML String
	 * 
	 * @see SemanticElementUtil#SEMANTIC_ELEMENT_KEY
	 * @see SemanticElementUtil#ADAPTER_KEY
	 * @see SemanticElementUtil#ADAPTER_FACTORY_KEY
	 */
	public static Map<String, Object> createSemanticElement(String xmlString,
	        String semanticElementId) throws Exception {
        DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
        DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
        Document doc = dBuilder.parse(new ByteArrayInputStream(xmlString.getBytes("UTF-8")));
        Element rootElement = doc.getDocumentElement();

        SemanticElementFactory seFactory = new SemanticElementFactory(Utility.TEMPLATE_EDITOR_ID);
        SemanticElement element = seFactory.createById(semanticElementId);

        XmlAdapterFactory adapterFactory = new SimpleDOMAdapterFactory(doc, seFactory);
        XmlAdapter adapter = adapterFactory.adapt(rootElement);
        adapter.initialize(element);
        if (element instanceof NodeElementContainer)
            setTransitionSources((NodeElementContainer) element);

        Map<String, Object> output = new HashMap<String, Object>();
        output.put(ADAPTER_KEY, adapter);
        output.put(ADAPTER_FACTORY_KEY, adapterFactory);
        output.put(SEMANTIC_ELEMENT_KEY, element);

        return output;
	}

	/**
	 * Creates a Workflow Template Semantic Element from the Xml String.
	 * 
	 * @param xmlString
	 *            The XML representation of the Workflow Template
	 * 
	 * @return The Workflow Template
	 * 
	 * @throws IOException
	 *             Failed to build the Model Element from the XML String
	 * @throws UnsupportedEncodingException
	 *             Failed to build the Model Element from the XML String
	 * 
	 * @see SemanticElementUtil#createSemanticElement(String, String, String)
	 */
	public static WorkflowTemplate createWorkflowTemplate(String xmlString)
			throws Exception {
		Map<String, Object> output = createSemanticElement(xmlString, TEMPLATE_SEID);
		WorkflowTemplate template = (WorkflowTemplate) output.get(SEMANTIC_ELEMENT_KEY);

		return template;
	}

	/**
	 * Creates a node definition semantic element from the specified XML
	 * 
	 * @param xml
	 * @return
	 * @throws IOException
	 */
	public static NodeDefinition createNodeDefinition(String xml) throws Exception {
	    Map<String, Object> output = createSemanticElement(xml, NODE_DEFN_SEID);
		NodeDefinition node = (NodeDefinition) output.get(SEMANTIC_ELEMENT_KEY);

		return node;
	}
	
	/**
	 * Gets the Semantic Element in the output map produced when a call is made to create a Semantic
	 * Element from the xml.
	 * 
	 * @param semanticElementOutputMap
	 *            The output produced from a call to createSemanticElement().
	 * 
	 * @return The Semantic Element present in the map.
	 * 
	 * @see SemanticElementUtil#createSemanticElement(String, String, String)
	 */
	public static SemanticElement getSemanticElement(Map<String, Object> semanticElementOutputMap)
	{
		return (SemanticElement) semanticElementOutputMap.get(SEMANTIC_ELEMENT_KEY);
	}

	/**
	 * Gets the Semantic Element Xml Adapter in the output map produced when a call is made to
	 * create a Semantic Element from the xml.
	 * 
	 * @param semanticElementOutputMap
	 *            The output produced from a call to createSemanticElement().
	 * 
	 * @return The Xml Adapter present in the map.
	 * 
	 * @see SemanticElementUtil#createSemanticElement(String, String, String)
	 */
	public static XmlAdapter getAdapter(Map<String, Object> semanticElementOutputMap)
	{
		return (XmlAdapter) semanticElementOutputMap.get(ADAPTER_KEY);
	}

	/**
	 * Gets the Semantic Element Xml Adapter Factory in the output map produced when a call is made
	 * to create a Semantic Element from the xml.
	 * 
	 * @param semanticElementOutputMap
	 *            The output produced from a call to createSemanticElement().
	 * 
	 * @return The Xml Adapter Factory present in the map.
	 * 
	 * @see SemanticElementUtil#createSemanticElement(String, String, String)
	 */
	public static XmlAdapterFactory getAdapterFactory(Map<String, Object> semanticElementOutputMap)
	{
		return (XmlAdapterFactory) semanticElementOutputMap.get(ADAPTER_FACTORY_KEY);
	}

	/*
	 * Sets the source for all transitions in the NodeElementContainer.
	 */
	private static void setTransitionSources(NodeElementContainer container){
		for(NodeElement node : container.getNodeElements()){
			for(Transition transition : node.getTransitions()){
				transition.setSource(node);
			}
		}
	}
}
