package com.tandbergtv.watchpoint.studio.validation.rules.nodedefinition.graph;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.graph.NodeDefinitionGraph;
import com.tandbergtv.watchpoint.studio.validation.graph.WatchPointGraphUtils;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

import edu.uci.ics.jung.graph.Vertex;

/**
 * Rule that validates that the Node Definition Graph contains only one root node.
 * 
 * @author Imran Naqvi
 *
 */
public class RequiredRootNodeRule extends ValidationRule<NodeDefinitionGraph> {
	/**
	 * Validates that the Graph element contains only one root node.
	 * 
	 * @param graph
	 *            The object being validated
	 * 
	 * @return The list of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	public List<ValidationMessage> validateRule(NodeDefinitionGraph graph)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();
		int rootNodeCount = 0;
		Set<Vertex> vertices = graph.getVertices();
		for (Vertex vertex : vertices)
		{
			if(vertex.inDegree() == 0)
				rootNodeCount++;
		}

		Object graphElement = WatchPointGraphUtils.getElement(graph);

		this.validateRootNodeCount(rootNodeCount, messages, graphElement);

		return messages;
	}
	

	/*
	 * Validate the Root Node Count (must equal 1).
	 */
	private void validateRootNodeCount(int count, List<ValidationMessage> messages, Object element)
	{
		/* Check if the Root node is missing or there are extra start nodes */
		if (count == 0)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.NODEDEFINITION_MISSING_ROOT_NODE);
		}
		else if (count > 1)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.NODEDEFINITION_EXTRA_ROOT_NODE);
		}
	}
}
