/**
 * NestingRule.java
 * Created Jan 13, 2012
 */
package com.tandbergtv.watchpoint.studio.validation.rules.processstate;

import static com.tandbergtv.watchpoint.studio.service.ServiceFactory.createFactory;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.TEMPLATE_SUBPROCESS_MORE_THAN_ONE_LEVEL_HIERARCHY;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.TEMPLATE_SUBPROCESS_NOT_VALID;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.jbpm.gd.jpdl.model.ProcessState;

import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Basic checks of the ProcessState node
 * 
 * @author Sahil Verma
 */
public class SubProcessRule extends ValidationRule<ProcessState> {

	/**
	 * {@inheritDoc}
	 */
	public List<ValidationMessage> validateRule(ProcessState node) {
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();
		List<WorkflowTemplateDTO> templates = getTemplates(node);
		
		if (templates.isEmpty()) {
			// Missing sub-process template
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, TEMPLATE_SUBPROCESS_NOT_VALID);
			return messages;
		}
		
		WorkflowTemplateDTO template = templates.get(0);

		if (template.isHasErrors()) {
			// Cannot use a template that has errors
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, TEMPLATE_SUBPROCESS_NOT_VALID);
			return messages;
		}
		
		if (template.getSubprocesses().isEmpty())
			return messages;
		
		// More than one level of hierarchy not allowed
		List<String> params = Arrays.asList(new String[] { template.getName() });
		ValidationMessageAdder.getInstance().addValidationMessage(messages, node, TEMPLATE_SUBPROCESS_MORE_THAN_ONE_LEVEL_HIERARCHY, params);

		return messages;
	}
	
	private List<WorkflowTemplateDTO> getTemplates(ProcessState node) {
		String templatename = node.getSubProcess().getName();

		if (templatename.indexOf(" - v") > 0)
			templatename = templatename.substring(0, templatename.indexOf(" - v"));

		IWorkflowTemplateService service = createFactory().createWorkflowTemplateService();
		
		return service.getTemplateByName(templatename);
	}
}
