package com.tandbergtv.watchpoint.studio.debugger.runtime.output;

import java.io.PrintWriter;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.jbpm.context.def.VariableAccess;
import org.jbpm.graph.def.Node;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.def.Transition;
import org.jbpm.graph.exe.ExecutionContext;
import org.jbpm.graph.node.Decision;
import org.jbpm.graph.node.StartState;
import org.jbpm.graph.node.TaskNode;
import org.jbpm.taskmgmt.def.Task;

import com.tandbergtv.watchpoint.studio.debugger.core.graph.Decision2;
import com.tandbergtv.watchpoint.studio.debugger.core.graph.Loop2;
import com.tandbergtv.watchpoint.studio.debugger.core.graph.SuperState2;
import com.tandbergtv.watchpoint.studio.debugger.model.SimulationType;
import com.tandbergtv.watchpoint.studio.debugger.runtime.exception.NodeSimulationException;
import com.tandbergtv.watchpoint.studio.debugger.runtime.nodesimulation.MailNodeSimulator;
import com.tandbergtv.watchpoint.studio.debugger.runtime.nodesimulation.NodeSimulator;
import com.tandbergtv.workflow.core.MailNode;

public class StringToOutputStreamSimulationWriter extends TemplateSimulationOutputWriter {

	protected PrintWriter out;
	
	private static final String SECTION_SEPARATOR = "----------------------------------------------------";
	
	private static final String IDENT = "    ";
	
	public StringToOutputStreamSimulationWriter(PrintWriter printWriter) {
		this.out = printWriter;
	}

	@Override
	public void templateSimulationStarted(ExecutionContext context, ProcessDefinition template) {
		out.println(SECTION_SEPARATOR);
		out.println("Starting simulation for: " + template.getName());
		out.println(SECTION_SEPARATOR);
		out.flush();
	}

	@Override
	public void templateSimulationCompleted(ExecutionContext context, ProcessDefinition template) {
		out.println();
		out.println(SECTION_SEPARATOR);
		out.println("Ending simulation for: " + template.getName());
		Map<?, ?> variables = context.getContextInstance().getVariables(context.getToken());
		printVariables(variables);
		out.println(SECTION_SEPARATOR);
		out.close();
	}

	@Override
	public void nodeSimulationStarted(ExecutionContext context, Node node, NodeSimulator simulator) {
		out.println();
		out.println(SECTION_SEPARATOR);
		if (node.getSuperState() instanceof SuperState2) {
			node = node.getSuperState();
		}
		out.println("Node: " + node.getName());
		out.println(SECTION_SEPARATOR);
		
		if (simulator != null && simulator.getSimulationConfig() != null) {
			out.print(getIdentation(1));
			out.println("Simulation Type: " + simulator.getSimulationConfig().getSimulationType());
			
			printNodeSpecificData(context, node, simulator);
			if (! (node instanceof StartState) ) {
				printInputVariables(context, simulator.getVariables());
			}
		}
		out.flush();
	}
	
	@Override
	public void nodeSimulationCompleted(ExecutionContext context, Node node, NodeSimulator simulator) {
		if (simulator != null && simulator.getSimulationConfig() != null && !SimulationType.SKIP_NODE.equals(simulator.getSimulationConfig().getSimulationType())) {
			if (simulator.getVariables() != null && !simulator.getVariables().isEmpty()) {
				printOutputVariables(context, simulator.getVariables());
			}
			if (simulator.fail()) {
				printFail(simulator.getException());
			}
		}
		List<?> transitions = node.getLeavingTransitions();
		if (transitions != null && transitions.size() > 1) {
			if (simulator != null) {
				transitions = simulator.getLeavingTransitions();
			}
			if (node instanceof Decision) {
				printTransitions(transitions);
			} else if (transitions != null && transitions.size() > 1) {
				printTransitions(transitions);
			}
		}
		if (node instanceof Loop2) {
			Loop2 loop = (Loop2) node;
			out.print(getIdentation(1));
			out.print("Index: ");
			out.println(context.getVariable(loop.getIndex()));
			out.print(getIdentation(1));
			out.print("Loop ?: ");
			out.println(loop.evaluateExpression(context));
		}
		if (node instanceof MailNode) {
			MailNodeSimulator mailSimulator = (MailNodeSimulator) simulator;
			out.print("To: ");
			out.println(mailSimulator.getTo());
			out.print(getIdentation(1));
			out.print("Subject: ");
			out.println(mailSimulator.getSubject());
			out.print(getIdentation(1));
			out.print("Body: ");
			out.println(mailSimulator.getText());
			out.print(getIdentation(1));
			out.print("Attachment: ");
			out.println(mailSimulator.getAttachment());
		}
		out.println(SECTION_SEPARATOR);
		out.flush();
	}
	
	protected void printNodeSpecificData(ExecutionContext context, Node node, NodeSimulator simulator) {
		if (SimulationType.EXECUTE.equals(simulator.getSimulationConfig().getSimulationType())) {
			out.print(getIdentation(1));
			if (node instanceof Decision2) {
				Decision2 decision = (Decision2) node;
				out.print("Executing expression: ");
				out.println(decision.getDecisionExpression());
			}
			if (node instanceof Loop2) {
				Loop2 loop = (Loop2) node;
				out.print("Executing expression: ");
				out.println(loop.getExpression());
			}
			if (node instanceof TaskNode) {
				TaskNode taskNode = (TaskNode) node;
				Task task = node.getProcessDefinition().getTaskMgmtDefinition().getTask(node.getName());
				if (task == null) {
					out.print("Executing class: ");
					out.println(taskNode.getAction().getActionDelegation().getClassName());
				}
			}
		}
	}
	
	protected void printFail(NodeSimulationException nodeSimulationException) {
		out.print(getIdentation(1));
		if (nodeSimulationException == null) {
			out.println("Fail: true. Aborting execution.");
		} else {
			out.println();
			out.println("An error occurred while attempting to simulate node: " + nodeSimulationException.getNode().getName());
			out.println("Error: " + nodeSimulationException.getMessage());
			if (nodeSimulationException.getCause() != null) {
				out.println("Root cause: " + nodeSimulationException.getCause().getMessage());
			}
			out.println("StackTrace: ");
			nodeSimulationException.printStackTrace(out);
			out.println();
		}
	}
	
	protected void printTransitions(List<?> transitions) {
		for (Object t : transitions) {
			out.print(getIdentation(1));
			Transition transition = (Transition) t; 
			out.println("Transition: " + transition.getName());
		}
	}
	
	protected void printVariable(int index, Object varName, Object varValue) {
		out.print(getIdentation(2));
		out.print(index + ". ");
		out.print(varName);
		out.print("=");
		if (varValue != null)
			out.println(varValue);
		else
			out.println();
	}
	
	protected void printVariables(Map<?, ?> variables) {
		if (variables != null && !variables.isEmpty()) {
			out.print(getIdentation(1));
			out.println("Variables: ");
			int i = 1;
			for (Object var : variables.keySet()) {
				printVariable(i, var, variables.get(var));
				i++;
			}
		}
	}
	
	protected void printInputVariables(ExecutionContext context, Collection<VariableAccess> variables) {
		if (!variables.isEmpty()) {
			out.print(getIdentation(1));
			out.println("Input Variables: ");
			int i = 1;
			for (VariableAccess var : variables) {
				if (var.isReadable()) {
					printVariable(i, var.getMappedName(), context.getVariable(var.getVariableName()));
					i++;
				}
			}
		}
	}
	
	protected void printOutputVariables(ExecutionContext context, Collection<VariableAccess> variables) {
		if (!variables.isEmpty()) {
			out.print(getIdentation(1));
			out.println("Output Variables: ");
			int i = 1;
			for (VariableAccess var : variables) {
				if (var.isWritable()) {
					printVariable(i, var.getVariableName(), context.getVariable(var.getVariableName()));
					i++;
				}
			}
		}
	}
	
	protected void printVariables(ExecutionContext context, Collection<VariableAccess> variables) {
		out.print(getIdentation(1));
		out.println("Variables: ");
		int i = 1;
		for (VariableAccess var : variables) {
			printVariable(i, var.getVariableName(), context.getVariable(var.getVariableName()));
			i++;
		}
	}
	
	private String getIdentation(int ammount) {
		StringBuilder builder = new StringBuilder(ammount);
		for (int i = 0; i < ammount; i++) {
			builder.append(IDENT);
		}
		return builder.toString();
	}
	
	
}
