/*
 * Created on Sep 25, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart;

import java.util.Map;

import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.widgets.FormToolkit;

import com.tandbergtv.watchpoint.studio.dto.AdaptorScope;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.eventhandler.FormPartDirtyModifyListener;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.eventhandler.TextTrimFocusListener;

/**
 * The Custom Adaptor Section Part for the Resource Type
 * 
 * @author Vijay Silva
 */
public class CustomAdaptorSectionPart extends ResourceTypeSectionPart
{
	Label classNameLabel, parametersLabel;

	Text classNameText;

	Composite parametersComposite;

	ParametersCompositeContainer parametersContainer;

	// ========================================================================
	// ===================== CONSTRUCTORS
	// ========================================================================

	/**
	 * @param parentPart
	 *            The Part that contains this composite
	 * @param parent
	 *            The Parent Composite containing the section to be created
	 * @param toolkit
	 *            The toolkit to create the section with
	 * @param style
	 *            The style bits to create the section with
	 */
	public CustomAdaptorSectionPart(IEditorPart parentPart, Composite parent, FormToolkit toolkit,
			int style)
	{
		super(parentPart, parent, toolkit, style);
	}

	// ========================================================================
	// ===================== OVERRIDDEN SECTION PART METHODS
	// ========================================================================

	/**
	 * @see org.eclipse.ui.forms.AbstractFormPart#commit(boolean)
	 */
	@Override
	public void commit(boolean onSave)
	{
		super.commit(onSave);

		/* Update the Resource Type with the data in the widgets */
		ResourceType resourceType = this.getResourceType();
		if (resourceType != null)
		{
			String className = this.classNameText.getText();
			resourceType.setAdaptorClassName(className);
			resourceType.setAdaptorScope(AdaptorScope.APPLICATION);

			Map<String, String> parameters = this.parametersContainer.getParameters();
			resourceType.setAdaptorParameters(parameters);
		}
	}

	// ========================================================================
	// ===================== USER CONTROL INITIALIZATION
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#createSectionClientControls(org.eclipse.ui.forms.IManagedForm)
	 */
	@Override
	protected void createSectionClientControls(IManagedForm managedForm)
	{
		FormToolkit toolkit = managedForm.getToolkit();

		/* The Adaptor ClassName */
		this.classNameLabel = toolkit.createLabel(this.sectionClient, "Adaptor Class: ");
		this.classNameText = toolkit.createText(this.sectionClient, "");
		this.classNameText.setTextLimit(256);
		this.classNameText.addFocusListener(new TextTrimFocusListener());
		this.classNameText.addModifyListener(new FormPartDirtyModifyListener(this));

		/* The Adaptor Parameters */
		this.parametersLabel = toolkit.createLabel(this.sectionClient, "Adaptor Parameters: ");
		this.parametersComposite = toolkit.createComposite(this.sectionClient);
		toolkit.paintBordersFor(this.parametersComposite);
		this.parametersContainer = new ParametersCompositeContainer(this.editorPart, this, toolkit,
				this.parametersComposite);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#initializeSectionClientLayout(org.eclipse.ui.forms.IManagedForm)
	 */
	@Override
	protected void initializeSectionClientLayout(IManagedForm managedForm)
	{
		/* Set the Layout for the Section Client in the Form */
		this.sectionClient.setLayout(this.createGridLayout(2, false));

		/* Set the default Layout data for the widgets */
		this.classNameLabel.setLayoutData(this.createLabelGridData());
		this.parametersLabel.setLayoutData(this.createLabelGridData());

		/* Set the Layout data for the ClassName Text */
		GridData layoutData = this.createGridData();
		layoutData.widthHint = this.getDefaultClassNameTextWidth();
		this.classNameText.setLayoutData(layoutData);

		/* Set the Layout data for the Parameters Parent Composite */
		this.parametersComposite.setLayoutData(this.createGridData());
	}

	// ========================================================================
	// ===================== DISPLAYING THE INPUT DATA
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#populateSectionData()
	 */
	@Override
	protected void populateSectionData()
	{
		ResourceType resourceType = this.getResourceType();
		boolean isEditable = this.isInputEditable();

		if (resourceType != null)
		{
			String className = resourceType.getAdaptorClassName();
			this.classNameText.setText(this.formatValue(className, true));

			Map<String, String> adaptorParameters = resourceType.getAdaptorParameters();
			this.parametersContainer.setInput(adaptorParameters, isEditable);
		}
		else
		{
			this.classNameText.setText("");
			this.parametersContainer.setInput(null, isEditable);
		}
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#setWidgetPermissions()
	 */
	@Override
	protected void setWidgetPermissions()
	{
		boolean isEditable = this.isInputEditable();
		this.classNameText.setEditable(isEditable);
	}
}
