/**
 * ResourceTypeComparer.java
 * Created Apr 12, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.sync.compare;

import static com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants.RESOURCETYPE_SEID;
import static com.tandbergtv.watchpoint.studio.ui.util.Utility.TEMPLATE_EDITOR_ID;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.common.model.SemanticElementFactory;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITree;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITreeNode;
import com.tandbergtv.watchpoint.studio.ui.model.tree.provider.DefaultTree;
import com.tandbergtv.watchpoint.studio.ui.model.tree.provider.MutableTreeNode;
import com.tandbergtv.watchpoint.studio.ui.sync.DiffKind;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiffCalculator;
import com.tandbergtv.watchpoint.studio.ui.sync.impl.Diff;
import com.tandbergtv.watchpoint.studio.ui.sync.impl.UpdateElement;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

/**
 * Compares two resource types
 * 
 * @author Sahil Verma
 */
public class ResourceTypeComparer implements IDiffCalculator {
	
	private static final Logger logger = Logger.getLogger(ResourceTypeComparer.class);

	/**
	 * @param dto1
	 * @param dto2
	 * @return
	 */
	public ITree<IDiff> compare(IWatchPointDTO dto1, IWatchPointDTO dto2, IProgressMonitor monitor) {
		ResourceType local = ResourceType.class.cast(dto1);
		ResourceType remote = ResourceType.class.cast(dto2);

		ITree<IDiff> tree = createDiffTree(local, remote);
		ITreeNode<IDiff> root = tree.getRoot();
		
		try {
			monitor.beginTask("Calculating", 3);
			
			for (ITreeNode<IDiff> treenode : getAddedTreeNodes(local, remote))
				root.addChild(treenode);
			monitor.worked(1);
			
			for (ITreeNode<IDiff> treenode : getRemovedTreeNodes(local, remote))
				root.addChild(treenode);
			monitor.worked(1);
			
			for (ITreeNode<IDiff> treenode : getChangedTreeNodes(local, remote, new SubProgressMonitor(monitor, 1)))
				root.addChild(treenode);
		} finally {
			monitor.done();
		}

		logger.info(local.getName() + " " + root.getChildren().size() + " diffs");
		
		return tree;
	}
	
	/**
	 * Returns the tree nodes that contain a diff which represents a new node or message
	 * 
	 * @param local
	 * @param remote
	 * @return
	 */
	protected List<ITreeNode<IDiff>> getAddedTreeNodes(ResourceType local, ResourceType remote) {
		List<ITreeNode<IDiff>> treenodes = new ArrayList<ITreeNode<IDiff>>();
		
		for (Message message : getAddedMessages(local, remote))
			treenodes.add(new MutableTreeNode<IDiff>(new Diff(message, DiffKind.ADD)));
		
		for (NodeDefinitionDTO node : getAddedNodeDefinitions(local, remote)) {
			IDiff diff = new Diff(node, createSemanticElement(node), DiffKind.ADD);
			
			treenodes.add(new MutableTreeNode<IDiff>(diff));
		}

		return treenodes;
	}

	/**
	 * Returns the tree nodes that contain a diff which represents a deleted node or message
	 * 
	 * @param local
	 * @param remote
	 * @return
	 */
	protected List<ITreeNode<IDiff>> getRemovedTreeNodes(ResourceType local, ResourceType remote) {
		List<ITreeNode<IDiff>> treenodes = new ArrayList<ITreeNode<IDiff>>();

		for (NodeDefinitionDTO dto : getRemovedNodeDefinitions(local, remote)) {
			IDiff diff = new Diff(dto, createSemanticElement(dto), DiffKind.REMOVE);
			
			treenodes.add(new MutableTreeNode<IDiff>(diff));
		}

		for (Message message : getRemovedMessages(local, remote))
			treenodes.add(new MutableTreeNode<IDiff>(new Diff(message, DiffKind.REMOVE)));
		
		return treenodes;
	}
	
	/**
	 * Return the tree nodes containing diffs for modified node definitions
	 * 
	 * @param local
	 * @param remote
	 * @return
	 */
	protected List<ITreeNode<IDiff>> getChangedTreeNodes(ResourceType local, ResourceType remote,
		IProgressMonitor monitor) {
		List<ITreeNode<IDiff>> treenodes = new ArrayList<ITreeNode<IDiff>>();
		
		for (Message message : getChangedMessages(local, remote))
			treenodes.add(new MutableTreeNode<IDiff>(new Diff(message, DiffKind.CHANGE)));
		
		for (NodeDefinitionDTO node : local.getNodes()) {
			String name = node.getName();
			NodeDefinitionDTO n = remote.getNode(name);
			
			if (n != null) {
				n.setId(node.getId());
				n.setResourceType(local); /* Set the pointer to the new parent */
				ITreeNode<IDiff> treenode = getComparer(n).compare(node, n, monitor).getRoot();

				treenodes.add(treenode);
			}
		}
		
		return treenodes;
	}
	
	protected List<NodeDefinitionDTO> getAddedNodeDefinitions(ResourceType local, ResourceType remote) {
		List<NodeDefinitionDTO> nodes = new ArrayList<NodeDefinitionDTO>();
		
		for (NodeDefinitionDTO node : remote.getNodes()) {
			String name = node.getName();
			
			if (local.getNode(name) == null) {
				node.setResourceType(local); /* Set the pointer to the new parent */
				nodes.add(node);
			}
		}
		
		return nodes;
	}
	
	protected List<NodeDefinitionDTO> getRemovedNodeDefinitions(ResourceType local, ResourceType remote) {
		List<NodeDefinitionDTO> nodes = new ArrayList<NodeDefinitionDTO>();
		
		for (NodeDefinitionDTO node : local.getNodes()) {
			String name = node.getName();
			
			if (remote.getNode(name) == null)
				nodes.add(node);
		}
		
		return nodes;
	}
	
	/**
	 * Finds messages that are being added to the local resource type
	 * 
	 * @param local
	 * @param remote
	 * @return
	 */
	protected Collection<Message> getAddedMessages(ResourceType local, ResourceType remote) {
		Collection<Message> messages = new ArrayList<Message>();
		
		for (Message message : remote.getMessages()) {
			if (local.getMessage(message.getUid()) == null) {
				message.setResourceType(local); /* Set the pointer to the new parent */
				messages.add(message);
			}
		}
		
		return messages;
	}
	
	/**
	 * Finds messages that are being deleted from the local resource type
	 * 
	 * @param local
	 * @param remote
	 * @return
	 */
	protected Collection<Message> getRemovedMessages(ResourceType local, ResourceType remote) {
		Collection<Message> messages = new ArrayList<Message>();
		
		for (Message message : local.getMessages()) {
			if (remote.getMessage(message.getUid()) == null)
				messages.add(message);
		}
		
		return messages;
	}
	
	/**
	 * Finds messages in the given resource type that are being updated by the remote resource type
	 * 
	 * @param local
	 * @param remote
	 * @return
	 */
	protected Collection<Message> getChangedMessages(ResourceType local, ResourceType remote) {
		Collection<Message> messages = new ArrayList<Message>();
		
		for (Message message : local.getMessages()) {
			Message other = remote.getMessage(message.getUid());
			
			if (other != null && hasChanged(message, other)) {
				other.setResourceType(local);
				messages.add(other);
			}
		}
		
		return messages;
	}
	
	protected boolean hasChanged(Message local, Message remote) {
		if (!local.getName().equals(remote.getName()))
			return true;
		
		if (local.getDescription() == null && remote.getDescription() == null)
			return false;
		
		if (local.getDescription() != null)
			return !local.getDescription().equals(remote.getDescription());
		
		return true;
	}
	
	/**
	 * @return
	 */
	private IDiffCalculator getComparer(NodeDefinitionDTO node) {
		if (node.getType() == NodeDefinitionType.SuperState)
			return new SuperStateComparer();
		
		return new NodeDefinitionComparer();
	}
	
	protected NodeDefinition createSemanticElement(NodeDefinitionDTO dto) {
		NodeDefinition n = null;

		try {
			n = SemanticElementUtil.createNodeDefinition(dto.getXml());
			n.setId(dto.getId());
		} catch (Exception e) {
		}

		return n;
	}
	
	protected com.tandbergtv.watchpoint.studio.ui.model.ResourceType createSemanticElement(ResourceType local) {
		SemanticElement e = new SemanticElementFactory(TEMPLATE_EDITOR_ID).createById(RESOURCETYPE_SEID);
		com.tandbergtv.watchpoint.studio.ui.model.ResourceType rt = 
			(com.tandbergtv.watchpoint.studio.ui.model.ResourceType) e;
		
		rt.setId(local.getId());
		rt.setName(local.getName());
		
		return rt;
	}
	
	/**
	 * Creates the diff tree with the resource type as the root node
	 * 
	 * @param local
	 * @return
	 */
	protected ITree<IDiff> createDiffTree(ResourceType local, ResourceType remote) {
		IDiff diff = new UpdateElement(local, remote, createSemanticElement(remote), DiffKind.CHANGE);
		ITreeNode<IDiff> root = new MutableTreeNode<IDiff>(diff);
		ITree<IDiff> tree = new DefaultTree<IDiff>(root);
		
		return tree;
	}
}
