/**
 * UpdateTemplateSuperStateCommand.java
 * Created May 11, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.sync.template;

import java.util.ArrayList;
import java.util.List;

import org.jbpm.gd.jpdl.model.NodeElement;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.IUpdateElement;
import com.tandbergtv.watchpoint.studio.ui.sync.command.AbstractDiffCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.util.ISynchronizationContext;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

/**
 * Updates a superstate node within a template
 * 
 * @author Sahil Verma
 */
public class UpdateTemplateSuperStateCommand extends AbstractDiffCommand {

	/**
	 * The new superstate
	 */
	protected NodeDefinitionDTO node;
	
	/**
	 * The new superstate
	 */
	protected NodeDefinition templateNode;
	
	/**
	 * Template model
	 */
	protected WorkflowTemplateDTO model;
	
	/**
	 * @param name
	 * @param diff
	 * @param context
	 */
	public UpdateTemplateSuperStateCommand(String name, IDiff diff, ISynchronizationContext context) {
		super(name, diff, context);
		IUpdateElement update = IUpdateElement.class.cast(diff);
		
		this.node = (NodeDefinitionDTO) update.getModel();
	}
	
	/**
	 * 		
	 * @param templateNode
	 * @param node
	 * @param container
	 */
	public UpdateTemplateSuperStateCommand(NodeDefinition templateNode, NodeDefinitionDTO node) {
		this.templateNode = templateNode;
		this.node = node;
	}


	/**
	 * @return the model
	 */
	public WorkflowTemplateDTO getModel() {
		return model;
	}

	/**
	 * @param model the model to set
	 */
	public void setModel(WorkflowTemplateDTO model) {
		this.model = model;
	}

	/**
	 * {@inheritDoc}
	 */
	public void execute() {
		/* Get all instances of the superstate in the template and modify their graph */
		replaceSuperStateChildren(templateNode);
	}
	
	/**
	 * Recreates the children of the given superstate in the template. Also preserves the IDs of the
	 * children.
	 * 
	 * @param superstate
	 */
	protected void replaceSuperStateChildren(NodeDefinition superstate) {
		/* Wipe out the existing superstate's children */
		for (NodeElement child : superstate.getNodeElements())
			superstate.removeNodeElement(child);
		
		/* Clone the superstate because we need to create a new reference before adding to the template */
		NodeDefinition clone = createSemanticElement();
		
		/* Set the definition in case the superstate node definition is in a template */
		if(superstate.getDefinitionName() != null){
			superstate.setDefinitionName(node.getName());
		}
		
		/* Now copy the children from the new superstate */ 
		for (NodeElement e : clone.getNodeElements())
			superstate.addNodeElement(e);
		
		// Remove all variables that are not present on the SuperState anymore
		List<WPVariable> toRemoveVars = new ArrayList<WPVariable>(); // to avoid concurrent modification exception
		for (WPVariable templateVariable : superstate.getVariables()) {
			if (!clone.hasVariable(templateVariable.getMappedName())) {
				toRemoveVars.add(templateVariable);
			}
		}
		for (WPVariable toRemove : toRemoveVars) {
			superstate.removeVariable(toRemove.getMappedName());
		}
		
		// Add the new variables
		for (WPVariable newVariable : clone.getVariables()) {
			WPVariable variable = superstate.getVariable(newVariable.getMappedName());
			if (variable == null) {
				superstate.addVariable(newVariable);
			} else {
				// Adjust the correct variable type...
				if (!variable.getType().equals(newVariable.getType())) {
					variable.setType(newVariable.getType());
				}
			}
		}
		
	}
	
	protected NodeDefinition createSemanticElement() {
		NodeDefinition element = null;

		try {
			element = SemanticElementUtil.createNodeDefinition(node.getXml());
			element.setId(node.getId());
			element.setResourceType(node.getResourceType().getId());
		} catch (Exception e) {
		}

		return element;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return "Template " + model.getName() + ", node " + node.getName() + " " + diff.getKind();
	}
}
