/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.util;

import java.io.IOException;
import java.util.InvalidPropertiesFormatException;
import java.util.List;
import java.util.Properties;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;

/**
 * Loads validation messages from a properties file.
 * 
 * @author Imran Naqvi
 * 
 */
public class ValidationMessages extends Properties
{

	private static final String PARAM_START = "<![";

	private static final String PARAM_END = "]>";

	/**
	 * 
	 */
	private static final long serialVersionUID = -9033237554511674020L;

	private static final String RESOURCE_PATH = "/ValidationMessages.properties";

	// Singleton Instance
	private static ValidationMessages instance;

	private ValidationMessages() throws InvalidPropertiesFormatException, IOException
	{
		super();
		this.loadProperties();
	}

	/**
	 * Singleton Accessor.
	 * 
	 * @return The Properties singleton instance
	 * @throws InvalidPropertiesFormatException
	 *             The Properties format is invalid
	 * @throws IOException
	 *             Failure to read the load the Properties
	 */
	public static synchronized ValidationMessages getInstance()
			throws InvalidPropertiesFormatException, IOException
	{
		if (instance == null)
		{
			instance = new ValidationMessages();
		}

		return instance;
	}

	public String getMessage(ValidationMessage validationMessage)
	{
		String code = validationMessage.getCode();
		final String message = this.getProperty(code);
		if (message == null)
			return "";

		int index = 0;
		StringBuilder buf = new StringBuilder();
		while (index < message.length())
		{

			/* find the PARAM_START and PARAM_END strings */
			int startIndex = message.indexOf(PARAM_START, index);
			int endIndex = (startIndex != -1) ? message.indexOf(PARAM_END, startIndex) : -1;

			if (startIndex != -1)
			{
				/* Append the message up to the PARAM_START */
				buf.append(message.substring(index, startIndex));

				if (endIndex != -1)
				{
					/* Found the PARAM_END */
					String parameter = message.substring(startIndex + PARAM_START.length(),
							endIndex);
					String value = this.getReplacementParameterValue(parameter, validationMessage);
					buf.append(value);
					index = endIndex + PARAM_END.length();
				}
				else
				{
					/* No PARAM_END, skip PARAM_START */
					index = startIndex + PARAM_START.length();
				}
			}
			else
			{
				/* No PARAM_START, go to end of the message */
				buf.append(message.substring(index, message.length()));
				index = message.length();
			}
		}

		return buf.toString();
	}

	/* Replace Parameter read from the Property with the value from the message parameter list */
	private String getReplacementParameterValue(String parameter, ValidationMessage message)
	{
		List<String> values = message.getParameterValues();
		if (values == null || parameter == null)
			return "";

		String value = parameter;
		int index = -1;
		try
		{
			index = Integer.parseInt(parameter);
		}
		catch (NumberFormatException nfe)
		{
			return value;
		}

		if (index >= 0 && index < values.size())
		{
			value = values.get(index);
			if (value == null)
				value = "";
		}
		else
		{
			value = "";
		}

		return value;
	}

	/*
	 * Initialize the Properties by loading the values from the Validation Messages Properties file.
	 */
	private void loadProperties() throws InvalidPropertiesFormatException, IOException
	{
		this.load(ValidationMessages.class.getResourceAsStream(RESOURCE_PATH));
	}
}
