package com.tandbergtv.watchpoint.studio.ui.wizard;

import static com.tandbergtv.watchpoint.studio.application.ApplicationPropertyKeys.WORKFLOW_SYSTEM_ID;
import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.SCHEMA_FILE_NAME;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.ArrayList;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;
import org.jbpm.gd.jpdl.wizard.NewResourceTypeProjectWizardCustom;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.tandbergtv.watchpoint.studio.application.ApplicationProperties;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer;
import com.tandbergtv.watchpoint.studio.util.ProjectPreferencesUtil;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;

/**
 * @author Patrik Araujo
 */
public class NewResourceTypeProjectWizard extends NewResourceTypeProjectWizardCustom {
	private static final String SCHEMA_IMPORT_ELEMENT = "xs:import";
	
	
	NewResourceTypeProjectWizardPage resourceTypePage;
	IResourceTypeService rtService;
	ResourceType resourceType;
	
	public NewResourceTypeProjectWizard() {
		super();
		rtService = ServiceFactory.createFactory().createResourceTypeService();
	}

	/**
	 * Get the Resource Type created by this wizard
	 * @return the resourceType
	 */
	public ResourceType getResourceType() {
		return this.resourceType;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.Wizard#performFinish()
	 */
	@Override
	public boolean performFinish() {
		try {
			if(!validateFormPage()){
				return false;
			}
			
			ResourceType newResourceType = this.resourceTypePage.getResourceTypeObject();
			
			this.resourceType = newResourceType;
			
			// Creates the java project related to the resource type 
			createJavaProject();
			selectAndReveal(super.getNewProject());
			
			// Refresh resource type explorer and resource group explorer
			refreshExtPointView("com.tandbergtv.gd.ResourceTypeExplorer");
			refreshExtPointView("com.tandbergtv.gd.ResourceGroup");
		}
		catch(Exception e) {
			Utility.reportError("Resource Type Creation Failed.", e);
			return false;
		}
		
		return true;
	}

	@Override
	public boolean validateFormPage() {
		/* Check if the name is unique */
		if(!this.rtService.isResourceTypeNameUnique(this.resourceTypePage.getResourceTypeName())) {
			this.resourceTypePage.setErrorMessage("A Resource Type with this name already exists");
			return false;
		}
		/* Check if the systemId is unique */
		if(!this.rtService.isResourceTypeSystemIdUnique(this.resourceTypePage.getResourceTypeSystemID())) {
			this.resourceTypePage.setErrorMessage("A Resource Type with this System ID already exists");
			return false;
		}
		/* Check if the systemId is equals to the workflow systemId */
		ApplicationProperties properties = ApplicationProperties.getInstance();
		String workflowSystemId = properties.getProperty(WORKFLOW_SYSTEM_ID);
		if(this.resourceTypePage.getResourceTypeSystemID().equals(workflowSystemId)){
			this.resourceTypePage.setErrorMessage("The System ID can't be the same as the Workflow System");
			return false;
		}
		
		return true;
	}

	@Override
	public void refreshExtPointView(String viewId) {
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		AbstractTreeViewExplorer explorer = null;
		try {
			IViewPart view = window.getActivePage().showView(viewId);
			explorer = (AbstractTreeViewExplorer) view;
			explorer.refresh();
		} catch (PartInitException e) {
			Utility.reportError("Impossible to refresh resource type view.", e);
		}
	}

	@Override
	public void addResourceTypePage() {
		resourceTypePage = new NewResourceTypeProjectWizardPage();
		addPage(resourceTypePage);
	}
	
	/**
	 * Creates the java project related to the resource type
	 */
	@Override
	public void createJavaProject() { 
		try {
			ResourceTypeJavaProjectCreator projectCreator = new ResourceTypeJavaProjectCreator(
					resourceType, resourceType.getName(), null);
			super.setNewProject(projectCreator.createJavaProject());
			createInitialContent(super.getNewProject());
		} catch (RuntimeException e) {
			Utility.reportError("Error while creating Resource Type project", e);
		}
	}

	@Override
	public void handleProjectException(String message, Exception e) {
		Utility.reportError("Resource Type Creation Failed.", e);
		try {
			if(getWorkspaceProject(super.getProjectName()).exists()){
				getWorkspaceProject(super.getProjectName()).delete(true, null);
			}
		} catch (CoreException ex) {
			ErrorDialog.openError(getShell(), "Problem deleting java project", null, ex.getStatus());
		}
		
	}

	@Override
	public void copySchemaFile(IProject javaProject) throws Exception {
		File schemaFile = resourceTypePage.getFile();
		if(schemaFile != null){
			String schemaFileName = schemaFile.getName();
			// Create schema file
			createProjectFile(javaProject, schemaFileName, schemaFile);
			IFile file = javaProject.getProject().getFile(schemaFileName);
			try {
				ArrayList<String> importPaths = getSchemaImportFilePath(file.getLocation().toOSString());
				if(importPaths.size() > 0){
					for (String importPath : importPaths) {
						File schemaImportFile = getImportFile(schemaFile, importPath);
						if(schemaImportFile != null){
							// Copy each import schema into the project
							createProjectFile(javaProject, schemaImportFile.getName(), schemaImportFile);
						}else{
							MessageDialog.openError(Display.getCurrent().getActiveShell(), "Problem",
							"Impossible to copy schema import file.\n" +"File path : " + importPath);
						}
					}
					updateSchemaImports(file);
				}
				// Set the schema file name in the project.
				ProjectPreferencesUtil.addProjectPreference(javaProject.getProject(), SCHEMA_FILE_NAME,
						schemaFileName);
			} catch (FileNotFoundException e) {
				ErrorDialog.openError(getShell(), "Problem creating schema files", null, null);	
			}
		}
	}

	@Override
	public ArrayList<String> getSchemaImportFilePath(String filePath) throws Exception {
		Document schema = XMLDocumentUtility.loadFile(new FileInputStream(filePath), false);
		NodeList importElements = schema.getElementsByTagName(SCHEMA_IMPORT_ELEMENT);
		ArrayList<String> filePaths = new ArrayList<String>();
		for (int i = 0; i < importElements.getLength(); i++) {
			Node importLocation = importElements.item(i).getAttributes().getNamedItem("schemaLocation");
			String importPath = importLocation.getNodeValue();
			filePaths.add(importPath);
		}
		return filePaths;
	}

	@Override
	public void updateSchemaImports(IFile file) throws Exception {
		Document schema = XMLDocumentUtility.loadFile(file.getContents(), false);
		NodeList importElements = schema.getElementsByTagName(SCHEMA_IMPORT_ELEMENT);
		for (int i = 0; i < importElements.getLength(); i++) {
			Node importLocation = importElements.item(i).getAttributes().getNamedItem("schemaLocation");
			File temp = new File(importLocation.getNodeValue());
			importLocation.setNodeValue(temp.getName());
		}
		ByteArrayOutputStream baos = XMLDocumentUtility.createByteArray(schema);
		byte[] contentArray = baos.toByteArray();
		file.setContents(new ByteArrayInputStream(contentArray), IFile.KEEP_HISTORY, null);
	}

}
