/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.wizard;

import static com.tandbergtv.watchpoint.studio.dto.IPersistable.DEFAULT_ID;
import static com.tandbergtv.watchpoint.studio.service.ServiceFactory.createFactory;
import static com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants.NODE_DEFN_SEID;
import static com.tandbergtv.watchpoint.studio.ui.util.ServiceErrors.getInstance;
import static com.tandbergtv.watchpoint.studio.ui.util.Utility.NODEDEF_EDITOR_ID;
import static com.tandbergtv.watchpoint.studio.ui.util.Utility.createNodeDefinitionDTO;
import static com.tandbergtv.watchpoint.studio.ui.util.Utility.getAdapterFromStream;
import static org.eclipse.jface.dialogs.MessageDialog.openError;
import static org.eclipse.swt.widgets.Display.getCurrent;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.jface.wizard.WizardPage;
import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.common.xml.XmlAdapter;
import org.w3c.dom.Document;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.ServiceException;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.service.ServiceValidationException;
import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * Wizard for creating a new superstate
 * 
 * @author Imran Naqvi
 * 
 */
public class NewSuperStateWizard extends AbstractWatchPointWizard {

	private static final String INITIAL_GPD_XML = "initialGPD.xml";

	private static final String INITIAL_SUPERSTATE_NODE_DEFINITION_XML = "initialSuperStateNodeDefinition.xml";
	
	private String resourceName;
	
	private ResourceType resourceType;

	private NewSuperStateWizardPage page;

	public void setResourceType(ResourceType resourceType) {
		this.resourceType = resourceType;
	}

	public ResourceType getResourceType() {
		return resourceType;
	}

	/**
	 * Ctor.
	 */
	public NewSuperStateWizard() {
		super("SuperState");
		setWindowTitle("New SuperState");
	}

	public void setResourceName(String resourceName) {
		this.resourceName = resourceName;
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.wizard.Wizard#addPages()
	 */
	public void addPages() {
		page = new NewSuperStateWizardPage(resourceName);
		addPage(page);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.tandbergtv.watchpoint.studio.ui.wizard.AbstractWatchPointWizard#isNameDuplicate(java.lang.String)
	 */
	@Override
	protected boolean isNameDuplicate(String name) {
		return ServiceFactory.createFactory().createNodeDefinitionService().nameExists(name);
	}

	/**
	 * Close all node definition editors.
	 */
	@Override
	protected void closeEditor() {
		Utility.closeEditor(Utility.NODEDEF_EDITOR_ID, true);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.tandbergtv.watchpoint.studio.ui.wizard.AbstractWatchPointWizard#getCurrentPage()
	 */
	@Override
	protected WizardPage getCurrentPage() {
		return this.page;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.tandbergtv.watchpoint.studio.ui.wizard.AbstractWatchPointWizard#setupEditor()
	 */
	@Override
	protected boolean setupEditor() {
		if (Utility.getWindow() == null)
			Utility.setWindow(getActiveWorkbenchWindow());
		
		try {
			NodeDefinitionDTO dto = this.createInitialDTO();
			
			dto.setType(NodeDefinitionType.SuperState);
			dto.setResourceType(getResourceType());

			this.saveNodeDefinition(dto);
		} catch (Exception e) {
			openError(getCurrent().getActiveShell(), "Error", "Error initializing superstate");
			return false;
		}

		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.tandbergtv.watchpoint.studio.ui.wizard.AbstractWatchPointWizard#getObjectNameFromPage()
	 */
	@Override
	protected String getObjectName() {
		return ((NewSuperStateWizardPage) this.page).getNodeDefinitionName();
	}

	/*
	 * Creates the initial superstate XML
	 * 
	 * @return
	 */
	private String createInitialNodeDefinition(String name, long id) {
		String xml = INITIAL_SUPERSTATE_NODE_DEFINITION_XML;
		SemanticElementFactory factory = new SemanticElementFactory(NODEDEF_EDITOR_ID);
		NodeDefinition node = (NodeDefinition) factory.createById(NODE_DEFN_SEID);
		XmlAdapter adapter = getAdapterFromStream(xml, this.getClass().getResourceAsStream(xml), factory);
		
		adapter.initialize(node);
		node.setName(name);
		node.setResourceType(id);
		
		return Utility.getXMLFromDocument(adapter.getNode().getOwnerDocument());
	}

	/*
	 * Creates the initial gpd xml
	 * 
	 * @return
	 */
	private String createInitialGpdInfo(String nodeDefName) throws Exception {
		DocumentBuilderFactory domFactory = DocumentBuilderFactory.newInstance();
	    domFactory.setNamespaceAware(true);
	    DocumentBuilder builder = domFactory.newDocumentBuilder();
	    Document doc = builder.parse(this.getClass().getResourceAsStream(INITIAL_GPD_XML));
	    
	    return Utility.getXMLFromDocument(doc);
	}

	/*
	 * Creates the initial NodeDefinitionDTO object.
	 * 
	 * @return initial NodeDefinitionDTO object
	 */
	private NodeDefinitionDTO createInitialDTO() throws Exception {
		String name = getObjectName();
		String xml = createInitialNodeDefinition(name, getResourceType().getId());
		String gpd = createInitialGpdInfo(name);
		
		return createNodeDefinitionDTO(DEFAULT_ID, name, xml, gpd);
	}

	/*
	 * Saves the superstate
	 * 
	 * @param dto the node definition to save
	 */
	private void saveNodeDefinition(NodeDefinitionDTO dto) {
		try {
			createFactory().createNodeDefinitionService().createNodeDefinitionInFileSystem(dto);
		} catch (ServiceValidationException e) {
			throw new UIException("The superstate is invalid, can not save.", e);
		} catch (ServiceException e) {
			throw new UIException(getInstance().getProperty(e.getServiceErrorCode().getCode()), e);
		} 
	}
}
