/*
 * Created on Nov 15, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.actionvariable;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import com.tandbergtv.watchpoint.studio.application.ApplicationProperties;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;
import com.tandbergtv.watchpoint.studio.validation.model.ActionVariable;

/**
 * Rule that validates an Action Variable ensuring that the name specified is valid.
 * 
 * @author Vijay Silva
 */
public class ActionVariableNameRule extends ValidationRule<ActionVariable>
{
	private static final String VALID_TAG_NAME_REGEX = "[a-zA-Z]\\p{Alnum}*";
	
	private static final String PROP_RESTRICTED_VAR_NAMES = "AutoTask.Action.RestrictedVarNames";

	/**
	 * Validates that the Name of the Action Variable is valid.
	 * 
	 * @param target
	 *            The Action Variable to validate
	 * @return The list of validation messages
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(ActionVariable target)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		// validate that the name is not blank
		this.validateBlankName(target, messages);

		// validate the value of the name
		this.validateNameValue(target, messages);

		return messages;
	}

	/*
	 * Validate if the Name is Blank.
	 */
	private void validateBlankName(ActionVariable target, List<ValidationMessage> messages)
	{
        if (this.isBlank(target.getElement().getName())) {
            ValidationMessageAdder.getInstance().addValidationMessage(messages, target.getElement(), ValidationMessageCode.ACTION_VARIABLE_NAME_BLANK);
        }
	}

	/*
	 * Validate the value of the Name.
	 */
	private void validateNameValue(ActionVariable target, List<ValidationMessage> messages)
	{
		if (this.isBlank(target.getElement().getName()))
			return;

		if (!Pattern.matches(VALID_TAG_NAME_REGEX, target.getElement().getName()))
		{
			ValidationMessageCode code = ValidationMessageCode.ACTION_VARIABLE_NAME_INVALID;
			List<String> parameters = new ArrayList<String>();
			parameters.add(target.getElement().getName());
			ValidationMessageAdder.getInstance().addValidationMessage(messages, target.getElement(), code, parameters);
		}

		if (restricted(target.getElement().getName()))
		{
			ValidationMessageCode code = ValidationMessageCode.ACTION_VARIABLE_NAME_NOT_ALLOWED;
			List<String> parameters = new ArrayList<String>();
			parameters.add(target.getElement().getName());
			ValidationMessageAdder.getInstance().addValidationMessage(messages, target.getElement(), code, parameters);
		}
	}
	
	// Checks if the given variable name is configured as not allowed.
	private boolean restricted(String varName)
	{
		String restrictedVarNames = ApplicationProperties.getInstance().getProperty(PROP_RESTRICTED_VAR_NAMES);
		if(restrictedVarNames == null || restrictedVarNames.trim().length() == 0)
			return false;
		for(String restrictedVarName : restrictedVarNames.split(","))
		{
			if(varName.equalsIgnoreCase(restrictedVarName.trim()))
				return true;
		}
		return false;
	}

	/* Method to check if the value is blank or null */
	private boolean isBlank(String value)
	{
		return (value == null || value.trim().length() == 0);
	}
}
