/*
 * Created on Oct 3, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.message;

import static com.tandbergtv.watchpoint.studio.application.ApplicationPropertyKeys.WORKFLOW_SYSTEM_ID;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import com.tandbergtv.watchpoint.studio.application.ApplicationProperties;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Validation Rule that validates the properties of a Message
 * 
 * @author Vijay Silva
 */
public class MessagePropertiesRule extends ValidationRule<Message>
{
	private static final String MESSAGE_UID_REGEX = "^(\\p{Alnum}{6})((\\.(\\p{Alnum}{1,3}))?)$";
	/**
	 * Validates the Message
	 * 
	 * @param message
	 *            The Message to validate
	 * 
	 * @return The list of Validation Messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(Message message)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		this.validateParentResourceType(message, messages);
		this.validateMessageName(message, messages);
		this.validateMessageUID(message, messages);
		this.validateMessageProtocol(message, messages);

		return messages;
	}

	/*
	 * Validates that the parent Resource Type is present
	 */
	private void validateParentResourceType(Message message, List<ValidationMessage> messages)
	{
		ResourceType resourceType = message.getResourceType();
		if (resourceType == null)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, message, ValidationMessageCode.RESOURCE_TYPE_MESSAGE_RESOURCE_TYPE_NULL);
		}
	}

	/*
	 * Validates that the Message name
	 */
	private void validateMessageName(Message message, List<ValidationMessage> messages)
	{
		String name = message.getName();
		if (name == null || name.trim().length() == 0)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, message, ValidationMessageCode.RESOURCE_TYPE_MESSAGE_NAME_BLANK);
		}
		else if (name.length() > 50)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, message, ValidationMessageCode.RESOURCE_TYPE_MESSAGE_NAME_TOO_LONG);
		}
	}

	/*
	 * Validate the Message UID
	 */
	private void validateMessageUID(Message message, List<ValidationMessage> messages)
	{
		if (message.isUIDGenerationRequired() || message.getResourceType() == null)
			return;

		ResourceType resourceType = message.getResourceType();
		String uid = message.getUid();

		if (uid == null || uid.trim().length() == 0)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, message, ValidationMessageCode.RESOURCE_TYPE_MESSAGE_UID_BLANK);
		}
		else if (!Pattern.matches(MESSAGE_UID_REGEX, uid))
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, message, ValidationMessageCode.RESOURCE_TYPE_MESSAGE_UID_INVALID);
		}
		else
		{
			ApplicationProperties properties = ApplicationProperties.getInstance();

			String sourceId = uid.substring(0, 2);
			String destinationId = uid.substring(2, 4);
			String systemId = resourceType.getSystemId();
			String wfsSystemId = properties.getProperty(WORKFLOW_SYSTEM_ID);

			if (message.isIncoming() && (!sourceId.equals(systemId)))
			{
				ValidationMessageAdder.getInstance().addValidationMessage(messages, message, ValidationMessageCode.INCOMING_MESSAGE_UID_SOURCE_INVALID);
			}

			if (message.isIncoming() && (!destinationId.equals(wfsSystemId)))
			{
				ValidationMessageAdder.getInstance().addValidationMessage(messages, message, ValidationMessageCode.INCOMING_MESSAGE_UID_DESTINATION_INVALID);
			}

			if ((!message.isIncoming()) && (!sourceId.equals(wfsSystemId)))
			{
				ValidationMessageAdder.getInstance().addValidationMessage(messages, message, ValidationMessageCode.OUTGOING_MESSAGE_UID_SOURCE_INVALID);
			}

			if ((!message.isIncoming()) && (!destinationId.equals(systemId)))
			{
				ValidationMessageCode code = ValidationMessageCode.OUTGOING_MESSAGE_UID_DESTINATION_INVALID;
				ValidationMessageAdder.getInstance().addValidationMessage(messages, message, code);
			}
		}
	}

	/*
	 * Validate the Message protocol
	 */
	private void validateMessageProtocol(Message message, List<ValidationMessage> messages)
	{
		String protocol = message.getProtocol();
		if (protocol != null && protocol.length() > 256)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, message, ValidationMessageCode.RESOURCE_TYPE_MESSAGE_PROTOCOL_TOO_LONG);
		}
	}

}
