/*
 * Created on Jun 25, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dataaccess.jpa;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.persistence.Query;

import com.tandbergtv.watchpoint.studio.dataaccess.ResourceGroupDAI;
import com.tandbergtv.watchpoint.studio.dto.ProtectionKey;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;

/**
 * The JPA Data Access Implementation for the ResourceGroup entity.
 * 
 * @author Vijay Silva
 */
public class ResourceGroupPDAO extends PersistenceDAO<ResourceGroup, Long> implements
		ResourceGroupDAI
{
	/**
	 * Constructor
	 * 
	 * @param persistenceContext
	 *            The Persistence Context to use for data operations.
	 */
	public ResourceGroupPDAO(JPAPersistenceContext persistenceContext)
	{
		super(ResourceGroup.class, persistenceContext);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceGroupDAI#findByPath(String)
	 */
	public ResourceGroup findByPath(String path) {
		String queryName = "ResourceGroup.ByPath";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("path", path);
		
		ResourceGroup result = null;
		try {
			result = (ResourceGroup) query.getSingleResult();
		} catch (javax.persistence.NoResultException e) {
			// No ResourceGroup found - return null
		}
		
		return result;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceGroupDAI#findByResourceTypeSystemId(java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	public List<ResourceGroup> findByResourceTypeSystemId(String systemId)
	{
		String queryName = "ResourceGroup.ByResourceTypeSystemId";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("systemId", systemId);
		return query.getResultList();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceGroupDAI#findByNames(Collection)
	 */
	public List<ResourceGroup> findByNames(Collection<String> names)
	{
		if (names == null || names.isEmpty())
			return new ArrayList<ResourceGroup>();

		// Build the Query String
		StringBuilder queryBuf = new StringBuilder();
		queryBuf.append("SELECT rg.id, rg.name ");
		queryBuf.append(this.getFromClause()).append(" rg ");
		queryBuf.append("WHERE rg.name IN ( ");

		boolean first = true;
		for (String name : names)
		{
			if (!first)
				queryBuf.append(", ");
			else
				first = false;

			queryBuf.append("'" + name + "'");
		}

		queryBuf.append(" ) ");

		// Create and execute the query
		Query query = this.getEntityManager().createQuery(queryBuf.toString());
		return this.buildResourceGroups(query.getResultList());

	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceGroupDAI#findByName(java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	public List<ResourceGroup> findByName(String name)
	{
		String queryName = "ResourceGroup.ByName";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("name", name);
		return (List<ResourceGroup>) query.getResultList();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceGroupDAI#findProtectionKeyByName(java.lang.String)
	 */
	public ProtectionKey findProtectionKeyByName(String name)
	{
		String queryName = "ResourceGroup.ProtectionKeyByName";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("name", name);
		
		ProtectionKey result = null;
		try {
			result = (ProtectionKey) query.getSingleResult();
		} catch (javax.persistence.NoResultException nr) { }
		return result;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceGroupDAI#getAllFunctionalTypesForResourceType(long)
	 */
	public List<String> getAllFunctionalTypesForResourceType(String resourceTypeSystemId)
	{
		String queryName = "ResourceGroup.FunctionalTypeByResourceType";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("systemId", resourceTypeSystemId);
		return convertObjectListToStringList(query.getResultList(), true);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceGroupDAI#isNameExists(java.lang.String)
	 */
	public boolean isNameExists(String name)
	{
		String queryName = "ResourceGroup.CountByName";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("name", name);
		Long count = (Long) query.getSingleResult();

		return (count > 0);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.ResourceGroupDAI#getCountByResourceType(String)
	 */
	public int getCountByResourceType(String resourceTypeSystemId)
	{
		String queryName = "ResourceGroup.CountByResourceType";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("systemId", resourceTypeSystemId);
		Long count = (Long) query.getSingleResult();

		return count.intValue();
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/*
	 * Create a ResourceGroupDTO entity from the array of column values.
	 */
	private List<ResourceGroup> buildResourceGroups(List<?> data)
	{
		List<ResourceGroup> resourceGroups = new ArrayList<ResourceGroup>();

		if (data != null)
		{
			for (Object dataObject : data)
			{
				Object[] columns = (Object[]) dataObject;
				ResourceGroup resourceGroup = new ResourceGroup();
				for (int i = 0; i < columns.length; i++)
				{
					switch (i)
					{
						case 0:
							resourceGroup.setId((Long) columns[0]);
							break;
						case 1:
							resourceGroup.setName((String) columns[1]);
							break;
						case 2:
							resourceGroup.setDescription((String) columns[2]);
							break;
						case 3:
							resourceGroup.setFunctionalType((String) columns[3]);
							break;
						case 4:
							resourceGroup.setInternallyAcquired((Boolean) columns[4]);
							break;
						case 5:
							resourceGroup.setProtectionKey((ProtectionKey) columns[5]);
							break;
						case 6:
							resourceGroup.setSystemId((String) columns[6]);
							break;
					}
				}

				resourceGroups.add(resourceGroup);
			}
		}

		return resourceGroups;
	}

	/*
	 * Converts (casts) a list of objects to a list of String
	 */
	private List<String> convertObjectListToStringList(List<?> objList, boolean ignoreNulls)
	{
		List<String> result = new ArrayList<String>();
		if (objList != null)
		{
			for (Object obj : objList)
			{
				if (ignoreNulls)
				{
					if (obj != null)
						result.add((String) obj);
				}
				else
				{
					result.add((String) obj);
				}
			}
		}
		return result;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<ResourceGroup> findByProject(String project) {
		String queryName = "ResourceGroup.ByProject";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("projectName", project);
		return (List<ResourceGroup>) query.getResultList();
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public List<ResourceGroup> findByTemplateUsage(String templatePath) {
		String queryName = "ResourceGroup.ByTemplateUsage";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("templatePath", templatePath);
		return (List<ResourceGroup>) query.getResultList();
	}

}
