/**
 * ResourceTypeDAO.java
 * Created Feb 12, 2010
 */
package com.tandbergtv.watchpoint.studio.external.fs.dao;

import static com.tandbergtv.watchpoint.studio.application.ApplicationPropertyKeys.WORKFLOW_SYSTEM_ID;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.PLUGIN_NODE_FOLDER;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.PLUGIN_RESOURCE_FOLDER_NAME;
import static java.io.File.separator;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;

import com.tandbergtv.watchpoint.studio.application.ApplicationProperties;
import com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext;
import com.tandbergtv.watchpoint.studio.dataaccess.ResourceTypeDAI;
import com.tandbergtv.watchpoint.studio.dto.AdaptorScope;
import com.tandbergtv.watchpoint.studio.dto.AdaptorType;
import com.tandbergtv.watchpoint.studio.dto.ConnectionType;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.external.fs.NodeDefinitionParser;
import com.tandbergtv.watchpoint.studio.external.jpf.JPFContext;
import com.tandbergtv.watchpoint.studio.external.jpf.JPFPluginManager;
import com.tandbergtv.watchpoint.studio.external.jpf.model.AdaptorDescriptor;
import com.tandbergtv.watchpoint.studio.external.jpf.model.ManagementDescriptor;
import com.tandbergtv.watchpoint.studio.external.jpf.model.MessageDescriptor;
import com.tandbergtv.watchpoint.studio.external.jpf.model.ResourceTypeDescriptor;

/**
 * DAO for resource types using a JPF backend
 * 
 * @author Sahil Verma
 */
public class ResourceTypeDAO implements ResourceTypeDAI {

	private IPersistenceContext context;
	
	private static final Logger logger = Logger.getLogger(ResourceTypeDAO.class);
	
	/**
	 * @param context
	 */
	public ResourceTypeDAO(JPFContext context) {
		super();
		this.context = context;
	}

	/**
	 * {@inheritDoc}
	 */
	public List<ResourceType> findAllViewable() {
		return findAll();
	}

	/**
	 * {@inheritDoc}
	 */
	public List<ResourceType> findAll() {
		List<ResourceType> types = new ArrayList<ResourceType>();
		JPFPluginManager loader = JPFContext.class.cast(context).getPluginManager();
		Collection<ResourceTypeDescriptor> plugins = loader.getResourceTypeDescriptors();
		
		for (ResourceTypeDescriptor plugin : plugins)
			types.add(createResourceType(plugin));
		
		return types;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IPersistenceContext getPersistenceContext() {
		return this.context;
	}
	
	protected ResourceType createResourceType(ResourceTypeDescriptor plugin) {
		ResourceType rt = new ResourceType();
		
		rt.setName(plugin.getName());
		rt.setSystemId(plugin.getSystemId());
		rt.setDescription(plugin.getDescription());
		rt.setVersion(plugin.getVersion());
		rt.setCreateDate(plugin.getCreateDate());
		
		logger.debug("Resource type " + rt.getName());

		copyAdaptorDescriptor(rt, plugin.getAdaptor());
		copyMessages(rt, plugin.getMessages());
		copyManagement(rt, plugin.getManagement());
		copyNodes(rt, plugin.getLocation());
		
		return rt;
	}

	/**
	 * @param rt
	 */
	protected void copyNodes(ResourceType rt, String location) {
		String path = location + separator + PLUGIN_RESOURCE_FOLDER_NAME + separator + PLUGIN_NODE_FOLDER;
		File file = new File(path);
		
		if (!file.exists())
			return;
		
		NodeDefinitionParser parser = new NodeDefinitionParser(path, rt.getMessages());
		
		for (NodeDefinitionDTO node : parser.parse())
			rt.addNode(node);
	}
	
	/**
	 * Copy adaptor related stuff
	 * 
	 * @param rt
	 * @param ad
	 */
	protected void copyAdaptorDescriptor(ResourceType rt, AdaptorDescriptor ad) {
		if (ad == null)
			return;
		
		String className = ad.getAdaptorClass();
		
		if (className != null) {
			rt.setAdaptorType(AdaptorType.CUSTOM);
			rt.setAdaptorClassName(className);
			
			if (ad.getProperties() != null) {
				for (String key : ad.getProperties().keySet())
					rt.addAdaptorParameter(key, ad.getProperties().get(key));
			}
		} else {
			rt.setAdaptorType(AdaptorType.DEFAULT);
			rt.setMessageIdentificationClassName(ad.getMessageIdentifierClass());	
		}
		
		if (ad.getScope() != null)
			rt.setAdaptorScope(AdaptorScope.valueOf(ad.getScope().toString()));
	}
	
	/**
	 * Copy hb & init information
	 * 
	 * @param rt
	 * @param management
	 */
	protected void copyManagement(ResourceType rt, ManagementDescriptor management) {
		rt.setConnectionType(new ConnectionType(management.getConnectionType().ordinal(), management.getConnectionType().getDisplayName()));
		
		if (management.getHeartbeatStrategyClass() != null) {
			rt.setHeartbeatRequired(true);
			rt.setHeartbeatStrategyClass(management.getHeartbeatStrategyClass());
			
			ConnectionType c = new ConnectionType();
			
			c.setId(management.getHeartbeatConnectionType().ordinal());
			c.setName(management.getHeartbeatConnectionType().getDisplayName());
			
			rt.setHeartbeatConnectionType(c);
			
			if (management.getHeartbeatProperties() != null) {
				for (String name : management.getHeartbeatProperties().keySet())
					rt.addHeartbeatParameter(name, management.getHeartbeatProperties().get(name));
			}
		}
		
		if (management.getInitializationStrategyClass() != null) {
			rt.setInitializationRequired(true);
			rt.setInitializationStrategyClass(management.getInitializationStrategyClass());

			if (management.getInitializationProperties() != null) {
				for (String name : management.getInitializationProperties().keySet())
					rt.addInitializationParameter(name, management.getInitializationProperties().get(name));
			}
		}
	}

	/**
	 * Copy messages
	 * 
	 * @param rt
	 * @param messages
	 */
	protected void copyMessages(ResourceType rt, Map<String, MessageDescriptor> messages) {
		ApplicationProperties properties = ApplicationProperties.getInstance();
		String systemId = properties.getProperty(WORKFLOW_SYSTEM_ID);
		
		for (String key : messages.keySet()) {
			MessageDescriptor m = messages.get(key);
			Message message = new Message();
			
			message.setDescription(m.getDescription());
			message.setResourceType(rt);
			message.setUid(m.getUID());
			message.setName(m.getName());
			message.setIncoming(!m.getUID().startsWith(systemId)); /* Weird... */
			message.setProtocol(m.getProtocol());
			
			rt.addMessage(message);
			logger.debug("\t[" + m.getUID() + "]");
		}
		
		if (messages != null && messages.size() > 0)
			rt.setMessageSequenceNumber(getMessageSequenceNumber(rt.getMessages()) + 1);
	}
	
	/**
	 * Finds the max sequence number among the specified UIDs
	 * @param messages
	 * @return
	 */
	private int getMessageSequenceNumber(Set<Message> messages) {
		int sequence = 0;
		
		for (Message message : messages) {
			String uid = message.getUid();
			
			if (!uid.contains(".")) {
				int i = Integer.parseInt(uid.substring(uid.length() - 2));
				
				if (sequence < i)
					sequence = i;
			} else {
				/* FIXME Parse generated UIDs */
			}
		}
		
		return sequence;
	}

	/**
	 * {@inheritDoc}
	 */
	public ResourceType findByName(String name) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public ResourceType findBySystemId(String systemId) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public int getCountByName(String name) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public int getCountBySystemId(String systemId) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public List<Message> getMessagesById(long resourceTypeId) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public List<Message> getMessagesByUid(Set<String> uids) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isOutOfTheBox(long resourceTypeId) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public ResourceType create(ResourceType entity) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public void delete(Long key) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public ResourceType find(Long key) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public ResourceType update(ResourceType entity) {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<ResourceType> findByProjectName(String projectName) {
		throw new UnsupportedOperationException();
	}

	@Override
	public ResourceType findByPath(String path) {
		throw new UnsupportedOperationException();
	}
}
