/*
 * Created on Mar 28, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.external.jpf.validation;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.tandbergtv.watchpoint.studio.external.jpf.model.MessageDescriptor;
import com.tandbergtv.watchpoint.studio.external.jpf.model.ResourceTypeDescriptor;

/**
 * Utility class that performs validation of the Resource Type Descriptor
 * 
 * @author Vijay Silva
 */
public final class ResourceTypeDescriptorValidator
{
	// The Regular Expression to validate the System Id.
	private static final String SYSTEM_ID_REGEX = "\\p{Alnum}{2}";

	/*
	 * Cannot instantiate this constructor, utility class with static methods only.
	 */
	private ResourceTypeDescriptorValidator()
	{ // Cannot instantiate
	}

	/**
	 * Method to validate the ResourceTypeDescriptor object. Checks all the properties, and returns
	 * a list of error messages for each of the failures. Returns an empty list if valid.
	 * 
	 * @param descriptor
	 *            The ResourceTypeDescriptor to validate
	 * 
	 * @return The list of error messages (or empty list if valid).
	 */
	public static List<String> validate(ResourceTypeDescriptor descriptor)
	{
		List<String> result = new ArrayList<String>();

		validateSystemId(descriptor, result);
		validateName(descriptor, result);

		validateMessages(descriptor, result);
		validateAdaptor(descriptor, result);

		validateManagement(descriptor, result);

		return result;
	}

	/*
	 * Validate the System Id
	 */
	private static void validateSystemId(ResourceTypeDescriptor descriptor, List<String> result)
	{
		String systemId = descriptor.getSystemId();
		if (systemId == null)
		{
			String msg = "The System Id specified cannot be blank.";
			result.add(msg);
		}
		else if (!systemId.matches(SYSTEM_ID_REGEX))
		{
			String msg = "The System Id specified must be exactly 2 characters, "
					+ "and contain only alphanumeric values.";
			result.add(msg);
		}
	}

	/*
	 * Validate the Name
	 */
	private static void validateName(ResourceTypeDescriptor descriptor, List<String> result)
	{
		String name = descriptor.getName();
		if (name == null || name.length() == 0)
		{
			String msg = "The Resource Type Name cannot be blank.";
			result.add(msg);
		}
	}

	/*
	 * Validate the list of Messages stored in the descriptor
	 */
	private static void validateMessages(ResourceTypeDescriptor descriptor, List<String> result)
	{
		String systemId = descriptor.getSystemId();
		Map<String, MessageDescriptor> messages = descriptor.getMessages();
		if (messages != null)
		{
			for (MessageDescriptor message : messages.values())
			{
				validateMessage(systemId, message, result);
			}
		}
	}

	/*
	 * Method to validate the Message Descriptor
	 */
	private static void validateMessage(String systemId, MessageDescriptor message,
			List<String> result)
	{
		String name = message.getName();
		String uid = message.getUID();

		if (uid == null || uid.length() < 6)
		{
			String msg = "The Message UID: " + uid + " specified for message: " + name
					+ " is invalid.";
			result.add(msg);
		}

		String srcId = uid.substring(0, 2);
		String destId = uid.substring(2, 4);
		if (!(srcId.equals(systemId) || destId.equals(systemId)))
		{
			String msg = "The Message UID: " + uid + " specified for message: " + name
					+ " doesn't contain the System Id as the message source or destination, "
					+ "and is invalid.";
			result.add(msg);
		}

		if (name == null || name.length() == 0)
		{
			String msg = "The name of the message with UID: " + uid + " cannot be blank.";
			result.add(msg);
		}
	}

	/*
	 * Validate the list of Adaptor Properties
	 */
	private static void validateAdaptor(ResourceTypeDescriptor descriptor, List<String> result)
	{
	}

	/*
	 * Validate the Management properties
	 */
	private static void validateManagement(ResourceTypeDescriptor descriptor, List<String> result)
	{
	}
}
